<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\ContentEntityStorageInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\menu_revisions\Services\MenuHierarchyManager;
use Drupal\menu_revisions\Services\MenuRevisionManager;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Services\MenuRevisionManager
 * @group menu_revisions
 */
class MenuRevisionManagerGenerateMenuFromRevisionTest extends UnitTestCase {

  /**
   * The menu revision manager under test.
   *
   * @var \Drupal\menu_revisions\Services\MenuRevisionManager
   */
  protected $menuRevisionManager;

  /**
   * The mocked entity type manager.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $entityTypeManager;

  /**
   * The mocked database connection.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $database;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->prophesize(EntityTypeManagerInterface::class);

    // Set up other dependencies that aren't directly used in some tests.
    $currentUser = $this->prophesize(AccountProxyInterface::class);
    $this->database = $this->prophesize(Connection::class);
    $loggerFactory = $this->prophesize(LoggerChannelFactoryInterface::class);
    $logger = $this->prophesize(LoggerChannelInterface::class);
    $loggerFactory->get('menu_revision')->willReturn($logger->reveal());
    $hierarchyManager = $this->prophesize(MenuHierarchyManager::class);

    $this->menuRevisionManager = new MenuRevisionManager(
      $this->entityTypeManager->reveal(),
      $currentUser->reveal(),
      $this->database->reveal(),
      $loggerFactory->reveal(),
      $hierarchyManager->reveal()
    );
  }

  /**
   * @covers ::generateMenuFromRevision
   */
  public function testGenerateMenuFromRevisionNoMenuRevision() {
    $menu_id = 'main';
    $menu_revision_id = 10;

    // Mock the menu revision query to return no results.
    $menuRevisionQuery = $this->getMockBuilder(\stdClass::class)
      ->setMethods(['accessCheck', 'condition', 'execute'])
      ->getMock();
    $menuRevisionQuery->expects($this->any())->method('accessCheck')->willReturnSelf();
    $menuRevisionQuery->expects($this->any())->method('condition')->willReturnSelf();
    $menuRevisionQuery->expects($this->any())->method('execute')->willReturn([]);

    $menuRevisionStorage = $this->prophesize(EntityStorageInterface::class);
    $menuRevisionStorage->getQuery()->willReturn($menuRevisionQuery);
    $this->entityTypeManager->getStorage('menu_revision')->willReturn($menuRevisionStorage->reveal());

    $result = $this->menuRevisionManager->generateMenuFromRevision($menu_id, $menu_revision_id);

    $this->assertIsArray($result);
    $this->assertEmpty($result);
  }

  /**
   * @covers ::generateMenuFromRevision
   */
  public function testGenerateMenuFromRevisionBuildsTree() {
    $menu_id = 'main';
    $menu_revision_id = 10;

    // Mock the menu revision query to return a result (non-empty array).
    $menuRevisionQuery = $this->getMockBuilder(\stdClass::class)
      ->setMethods(['accessCheck', 'condition', 'execute'])
      ->getMock();
    $menuRevisionQuery->expects($this->any())->method('accessCheck')->willReturnSelf();
    $menuRevisionQuery->expects($this->any())->method('condition')->willReturnSelf();
    $menuRevisionQuery->expects($this->any())->method('execute')->willReturn([ $menu_revision_id => $menu_revision_id ]);

    $menuRevisionStorage = $this->prophesize(EntityStorageInterface::class);
    $menuRevisionStorage->getQuery()->willReturn($menuRevisionQuery);
    $this->entityTypeManager->getStorage('menu_revision')->willReturn($menuRevisionStorage->reveal());

    // Mock the database select -> execute -> fetchCol() chain to return two menu link revision ids.
    $select = $this->getMockBuilder(\stdClass::class)
      ->setMethods(['fields', 'condition', 'execute'])
      ->getMock();
    $select->expects($this->any())->method('fields')->willReturnSelf();
    $select->expects($this->any())->method('condition')->willReturnSelf();

    $executeResult = $this->getMockBuilder(\stdClass::class)
      ->setMethods(['fetchCol'])
      ->getMock();
    $executeResult->expects($this->any())->method('fetchCol')->willReturn([101, 102]);

    $select->expects($this->any())->method('execute')->willReturn($executeResult);

    $this->database->select('menu_revision_link', 'mrl')->willReturn($select);

    // Prepare two menu link revision entity mocks: parent and child.
    $parent = $this->getMockBuilder(\stdClass::class)
      ->setMethods(['uuid', 'getParentId', 'id', 'getTitle', 'getUrlObject', 'getWeight'])
      ->getMock();
    $parent->expects($this->any())->method('uuid')->willReturn('uuid-parent');
    $parent->expects($this->any())->method('getParentId')->willReturn(NULL);
    $parent->expects($this->any())->method('id')->willReturn(1);
    $parent->expects($this->any())->method('getTitle')->willReturn('Parent');
    $parent->expects($this->any())->method('getUrlObject')->willReturn('http://parent');
    $parent->expects($this->any())->method('getWeight')->willReturn(0);

    $child = $this->getMockBuilder(\stdClass::class)
      ->setMethods(['uuid', 'getParentId', 'id', 'getTitle', 'getUrlObject', 'getWeight'])
      ->getMock();
    $child->expects($this->any())->method('uuid')->willReturn('uuid-child');
    $child->expects($this->any())->method('getParentId')->willReturn('uuid-parent');
    $child->expects($this->any())->method('id')->willReturn(2);
    $child->expects($this->any())->method('getTitle')->willReturn('Child');
    $child->expects($this->any())->method('getUrlObject')->willReturn('http://child');
    $child->expects($this->any())->method('getWeight')->willReturn(0);

    // Mock the storage for menu_link_content to return the above revisions.
    $menuLinkStorage = $this->prophesize(ContentEntityStorageInterface::class);
    $menuLinkStorage->loadRevision(101)->willReturn($parent);
    $menuLinkStorage->loadRevision(102)->willReturn($child);
    $this->entityTypeManager->getStorage('menu_link_content')->willReturn($menuLinkStorage->reveal());

    // Execute the method under test.
    $result = $this->menuRevisionManager->generateMenuFromRevision($menu_id, $menu_revision_id);

    // Assertions: top-level has one parent item keyed by its uuid.
    $this->assertIsArray($result);
    $this->assertCount(1, $result);
    $this->assertArrayHasKey('uuid-parent', $result);

    $parent_item = $result['uuid-parent'];
    $this->assertEquals('Parent', $parent_item['title']);
    $this->assertArrayHasKey('below', $parent_item);
    $this->assertCount(1, $parent_item['below']);

    $child_item = $parent_item['below'][0];
    $this->assertEquals('Child', $child_item['title']);
    $this->assertEquals('uuid-child', $child_item['plugin_id']);
  }

}
