<?php

namespace Drupal\Tests\menu_revisions\Unit\Services;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\menu_revisions\Services\MenuHierarchyManager;
use Drupal\menu_revisions\Services\MenuRevisionManager;
use Drupal\Tests\UnitTestCase;
use Prophecy\PhpUnit\ProphecyTrait;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Services\MenuRevisionManager
 * @group menu_revisions
 */
class MenuRevisionManagerGetLatestActiveRevisionTest extends UnitTestCase {
  use ProphecyTrait;

  /**
   * The menu revision manager service.
   *
   * @var \Drupal\menu_revisions\Services\MenuRevisionManager
   */
  protected $menuRevisionManager;

  /**
   * The entity type manager prophecy.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $entityTypeManager;

  /**
   * The entity storage prophecy.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $entityStorage;

  /**
   * The query prophecy.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $query;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create prophecies for the dependencies.
    $this->entityTypeManager = $this->prophesize(EntityTypeManagerInterface::class);
    $this->entityStorage = $this->prophesize(EntityStorageInterface::class);
    $this->query = $this->prophesize(QueryInterface::class);
    $currentUser = $this->prophesize(AccountProxyInterface::class);
    $database = $this->prophesize(Connection::class);
    $loggerFactory = $this->prophesize(LoggerChannelFactoryInterface::class);
    $hierarchyManager = $this->prophesize(MenuHierarchyManager::class);

    // Set up the entity type manager to return our entity storage mock.
    $this->entityTypeManager->getStorage('menu_revision')
      ->willReturn($this->entityStorage->reveal());

    // Create the service with all its dependencies.
    $this->menuRevisionManager = new MenuRevisionManager(
      $this->entityTypeManager->reveal(),
      $currentUser->reveal(),
      $database->reveal(),
      $loggerFactory->reveal(),
      $hierarchyManager->reveal()
    );
  }

  /**
   * Tests the getLatestActiveMenuRevision method when active revisions exist.
   *
   * @covers ::getLatestActiveMenuRevision
   */
  public function testGetLatestActiveMenuRevisionWithResults() {
    $menuName = 'main';
    $expectedRevisionId = 42;

    // Configure the query prophecy to simulate finding a revision.
    $this->entityStorage->getQuery()
      ->willReturn($this->query->reveal());
    $this->query->accessCheck(TRUE)
      ->willReturn($this->query->reveal());
    $this->query->condition('menu_name', $menuName)
      ->willReturn($this->query->reveal());
    $this->query->condition('status', '1')
      ->willReturn($this->query->reveal());
    $this->query->sort('id', 'DESC')
      ->willReturn($this->query->reveal());
    $this->query->range(0, 1)
      ->willReturn($this->query->reveal());
    $this->query->execute()
      ->willReturn([$expectedRevisionId]);

    // Call the method and check the result.
    $result = $this->menuRevisionManager->getLatestActiveMenuRevision($menuName);
    $this->assertEquals($expectedRevisionId, $result);
  }

  /**
   * Tests the getLatestActiveMenuRevision method when no active revisions exist.
   *
   * @covers ::getLatestActiveMenuRevision
   */
  public function testGetLatestActiveMenuRevisionWithNoResults() {
    $menuName = 'main';

    // Configure the query prophecy to simulate no revisions found.
    $this->entityStorage->getQuery()
      ->willReturn($this->query->reveal());
    $this->query->accessCheck(TRUE)
      ->willReturn($this->query->reveal());
    $this->query->condition('menu_name', $menuName)
      ->willReturn($this->query->reveal());
    $this->query->condition('status', '1')
      ->willReturn($this->query->reveal());
    $this->query->sort('id', 'DESC')
      ->willReturn($this->query->reveal());
    $this->query->range(0, 1)
      ->willReturn($this->query->reveal());
    $this->query->execute()
      ->willReturn([]);

    // Call the method and check that the result is null.
    $result = $this->menuRevisionManager->getLatestActiveMenuRevision($menuName);
    $this->assertFalse($result);
  }

  /**
   * Tests the getLatestActiveMenuRevision method with multiple revisions.
   *
   * @covers ::getLatestActiveMenuRevision
   */
  public function testGetLatestActiveMenuRevisionWithMultipleResults() {
    $menuName = 'main';
    $revisionIds = [42, 37, 25]; // Sorted by ID descending

    // Configure the query prophecy to simulate multiple revisions.
    // Even though we use range(0, 1), we want to test that the function
    // correctly handles if multiple results were somehow returned.
    $this->entityStorage->getQuery()
      ->willReturn($this->query->reveal());
    $this->query->accessCheck(TRUE)
      ->willReturn($this->query->reveal());
    $this->query->condition('menu_name', $menuName)
      ->willReturn($this->query->reveal());
    $this->query->condition('status', '1')
      ->willReturn($this->query->reveal());
    $this->query->sort('id', 'DESC')
      ->willReturn($this->query->reveal());
    $this->query->range(0, 1)
      ->willReturn($this->query->reveal());
    $this->query->execute()
      ->willReturn($revisionIds);

    // Call the method and check that it returns the first (highest ID) revision.
    $result = $this->menuRevisionManager->getLatestActiveMenuRevision($menuName);
    $this->assertEquals(42, $result);
  }
}
