<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\menu_revisions\Entity\MenuRevision;
use Drupal\menu_revisions\Services\MenuHierarchyManager;
use Drupal\menu_revisions\Services\MenuRevisionManager;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Services\MenuRevisionManager
 * @group menu_revisions
 */
class MenuRevisionManagerGetLatestPublishedRevisionTest extends UnitTestCase {

  /**
   * The menu revision manager.
   *
   * @var \Drupal\menu_revisions\Services\MenuRevisionManager
   */
  protected $menuRevisionManager;

  /**
   * The entity type manager mock.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The entity storage mock.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $menuRevisionStorage;

  /**
   * The entity query mock.
   *
   * @var \Drupal\Core\Entity\Query\QueryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $query;

  /**
   * The current user mock.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $currentUser;

  /**
   * The database connection mock.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $database;

  /**
   * The logger channel mock.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $logger;

  /**
   * The logger factory mock.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $loggerFactory;

  /**
   * The menu hierarchy manager mock.
   *
   * @var \Drupal\menu_revisions\Services\MenuHierarchyManager|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $hierarchyManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mocks for all the dependencies.
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->menuRevisionStorage = $this->createMock(EntityStorageInterface::class);
    $this->query = $this->createMock(QueryInterface::class);
    $this->currentUser = $this->createMock(AccountProxyInterface::class);
    $this->database = $this->createMock(Connection::class);
    $this->logger = $this->createMock(LoggerChannelInterface::class);
    $this->loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $this->hierarchyManager = $this->createMock(MenuHierarchyManager::class);

    // Set up the logger factory to return our logger mock.
    $this->loggerFactory->expects($this->any())
      ->method('get')
      ->with('menu_revision')
      ->willReturn($this->logger);

    // Set up the entity type manager to return our storage mock.
    $this->entityTypeManager->expects($this->any())
      ->method('getStorage')
      ->with('menu_revision')
      ->willReturn($this->menuRevisionStorage);

    // Create the menu revision manager with mocked dependencies.
    $this->menuRevisionManager = new MenuRevisionManager(
      $this->entityTypeManager,
      $this->currentUser,
      $this->database,
      $this->loggerFactory,
      $this->hierarchyManager
    );
  }

  /**
   * Tests the getLatestPublishedRevision method.
   *
   * @covers ::getLatestPublishedRevision
   */
  public function testGetLatestPublishedRevision() {
    $menu_name = 'test_menu';
    $revision_id = 123;

    // Set up the query expectations for getLatestActiveMenuRevision.
    $this->menuRevisionStorage->expects($this->once())
      ->method('getQuery')
      ->willReturn($this->query);

    $this->query->expects($this->once())
      ->method('accessCheck')
      ->with(TRUE)
      ->willReturnSelf();

    $this->query->expects($this->exactly(2))
      ->method('condition')
      ->withConsecutive(
        ['menu_name', $menu_name],
        ['status', '1']
      )
      ->willReturnSelf();

    $this->query->expects($this->once())
      ->method('sort')
      ->with('id', 'DESC')
      ->willReturnSelf();

    $this->query->expects($this->once())
      ->method('range')
      ->with(0, 1)
      ->willReturnSelf();

    $this->query->expects($this->once())
      ->method('execute')
      ->willReturn([$revision_id]);

    // Create a mock menu revision entity.
    $menu_revision = $this->createMock(MenuRevision::class);

    // The storage should load the revision with the expected ID.
    $this->menuRevisionStorage->expects($this->once())
      ->method('load')
      ->with($revision_id)
      ->willReturn($menu_revision);

    // Call the method being tested.
    $result = $this->menuRevisionManager->getLatestPublishedRevision($menu_name);

    // Assert that the result is the expected menu revision.
    $this->assertSame($menu_revision, $result);
  }

  /**
   * Tests the getLatestPublishedRevision method when no published revisions exist.
   *
   * @covers ::getLatestPublishedRevision
   */
  public function testGetLatestPublishedRevisionNoResults() {
    $menu_name = 'test_menu';

    // Set up the query expectations for getLatestActiveMenuRevision.
    $this->menuRevisionStorage->expects($this->once())
      ->method('getQuery')
      ->willReturn($this->query);

    $this->query->expects($this->once())
      ->method('accessCheck')
      ->willReturnSelf();

    $this->query->expects($this->exactly(2))
      ->method('condition')
      ->willReturnSelf();

    $this->query->expects($this->once())
      ->method('sort')
      ->willReturnSelf();

    $this->query->expects($this->once())
      ->method('range')
      ->willReturnSelf();

    // Return empty array to simulate no published revisions.
    $this->query->expects($this->once())
      ->method('execute')
      ->willReturn([]);

    // Call the method being tested.
    $result = $this->menuRevisionManager->getLatestPublishedRevision($menu_name);

    // Assert that the result is NULL since there are no published revisions.
    $this->assertNull($result);
  }
}
