<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Database\StatementInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\Entity\RevisionableStorageInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Menu\MenuLinkInterface;
use Drupal\Core\Menu\MenuLinkManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\menu_revisions\Entity\MenuRevision;
use Drupal\menu_revisions\Services\MenuHierarchyManager;
use Drupal\menu_revisions\Services\MenuRevisionManager;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Services\MenuRevisionManager
 * @group menu_revisions
 */
class MenuRevisionManagerGetMenuTreeLinksTest extends UnitTestCase {

  /**
   * The menu revision manager service.
   *
   * @var \Drupal\menu_revisions\Services\MenuRevisionManager
   */
  protected $menuRevisionManager;

  /**
   * The mocked entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The mocked current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $currentUser;

  /**
   * The mocked database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $database;

  /**
   * The mocked logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $logger;

  /**
   * The mocked hierarchy manager.
   *
   * @var \Drupal\menu_revisions\Services\MenuHierarchyManager|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $hierarchyManager;

  /**
   * The mocked menu revision storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $menuRevisionStorage;

  /**
   * The mocked menu link content storage.
   *
   * @var \Drupal\Core\Entity\RevisionableStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $menuLinkStorage;

  /**
   * The mocked entity query.
   *
   * @var \Drupal\Core\Entity\Query\QueryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityQuery;

  /**
   * The mocked database select query.
   *
   * @var \Drupal\Core\Database\Query\SelectInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $selectQuery;

  /**
   * The mocked statement.
   *
   * @var \Drupal\Core\Database\StatementInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $statement;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mocks for dependencies.
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->currentUser = $this->createMock(AccountProxyInterface::class);
    $this->database = $this->createMock(Connection::class);
    $this->logger = $this->createMock(LoggerChannelInterface::class);
    $this->hierarchyManager = $this->createMock(MenuHierarchyManager::class);
    $this->menuRevisionStorage = $this->createMock(EntityStorageInterface::class);
    $this->menuLinkStorage = $this->createMock(RevisionableStorageInterface::class);
    $this->entityQuery = $this->createMock(QueryInterface::class);
    $this->selectQuery = $this->createMock(SelectInterface::class);
    $this->statement = $this->createMock(StatementInterface::class);

    $logger_factory = $this->createMock(LoggerChannelFactoryInterface::class);
    $logger_factory->method('get')->willReturn($this->logger);

    // Set up entity type manager storage mappings.
    $this->entityTypeManager->method('getStorage')
      ->willReturnMap([
        ['menu_revision', $this->menuRevisionStorage],
        ['menu_link_content', $this->menuLinkStorage]
      ]);

    // Configure entity query mock.
    $this->menuRevisionStorage->method('getQuery')
      ->willReturn($this->entityQuery);

    // Configure database select mock.
    $this->database->method('select')
      ->willReturn($this->selectQuery);

    // Set up Drupal container with menu link manager mock.
    $menu_link_manager = $this->createMock(MenuLinkManagerInterface::class);
    $container = $this->createMock('Symfony\Component\DependencyInjection\ContainerInterface');
    $container->method('get')
      ->with('plugin.manager.menu.link')
      ->willReturn($menu_link_manager);
    \Drupal::setContainer($container);

    // Create the menu revision manager instance.
    $this->menuRevisionManager = new MenuRevisionManager(
      $this->entityTypeManager,
      $this->currentUser,
      $this->database,
      $logger_factory,
      $this->hierarchyManager
    );
  }

  /**
   * Tests getMenuTreeLinksFromMenuRevision with valid menu revision.
   *
   * @covers ::getMenuTreeLinksFromMenuRevision
   */
  public function testGetMenuTreeLinksFromMenuRevisionWithValidRevision() {
    $menu_id = 'test_menu';
    $menu_revision_id = 123;
    $revision_ids = [10, 20, 30];

    // Mock the menu revision query.
    $this->entityQuery->expects($this->once())
      ->method('accessCheck')
      ->with(TRUE)
      ->willReturnSelf();

    $this->entityQuery->expects($this->exactly(2))
      ->method('condition')
      ->withConsecutive(
        ['id', $menu_revision_id],
        ['menu_name', $menu_id]
      )
      ->willReturnSelf();

    $this->entityQuery->expects($this->once())
      ->method('execute')
      ->willReturn([$menu_revision_id]);

    // Mock the database select query for menu revision links.
    $this->selectQuery->expects($this->once())
      ->method('fields')
      ->with('mrl', ['menu_link_revision_id'])
      ->willReturnSelf();

    $this->selectQuery->expects($this->once())
      ->method('condition')
      ->with('menu_revision_id', $menu_revision_id)
      ->willReturnSelf();

    $this->selectQuery->expects($this->once())
      ->method('execute')
      ->willReturn($this->statement);

    $this->statement->expects($this->once())
      ->method('fetchCol')
      ->willReturn($revision_ids);

    // Create mock menu link revisions.
    $menu_link_revisions = [];

    foreach ($revision_ids as $revision_id) {
      $menu_link_revision = $this->createMock(MenuLinkContent::class);
      $uuid = "uuid-{$revision_id}";
      $plugin_id = "menu_link_content:{$revision_id}";

      $menu_link_revision->method('uuid')->willReturn($uuid);
      $menu_link_revision->method('getPluginId')->willReturn($plugin_id);
      $menu_link_revision->method('getPluginDefinition')->willReturn([
        'id' => $plugin_id,
        'title' => "Menu Item {$revision_id}",
        'menu_name' => $menu_id,
        'url' => "http://example.com/{$revision_id}",
        'weight' => $revision_id,
        'expanded' => FALSE,
        'enabled' => TRUE,
        'parent' => '',
        'route_name' => '',
        'route_parameters' => [],
        'options' => [],
        'class' => 'Drupal\menu_link_content\Plugin\Menu\MenuLinkContent',
        'form_class' => 'Drupal\menu_link_content\Form\MenuLinkContentForm',
        'delete_form_class' => 'Drupal\menu_link_content\Form\MenuLinkContentDeleteForm',
        'provider' => 'menu_link_content'
      ]);

      $menu_link_revisions[$revision_id] = $menu_link_revision;
    }

    // Configure loadRevision calls.
    $this->menuLinkStorage->expects($this->exactly(count($revision_ids)))
      ->method('loadRevision')
      ->willReturnCallback(function($revision_id) use ($menu_link_revisions) {
        return $menu_link_revisions[$revision_id] ?? NULL;
      });

    // Create mock menu link plugin instances that will be returned by createInstance.
    $menu_link_plugins = [];
    foreach ($revision_ids as $revision_id) {
      $menu_link_plugin = $this->createMock(MenuLinkInterface::class);
      $menu_link_plugin->method('getOptions')->willReturn([]);
      $menu_link_plugin->expects($this->once())
        ->method('updateLink')
        ->with(['options' => ['overrideRevisionID' => $revision_id]], FALSE);
      $menu_link_plugins[] = $menu_link_plugin;
    }

    // Mock the menu link manager service.
    $menu_link_manager = $this->createMock(MenuLinkManagerInterface::class);
    $menu_link_manager->expects($this->exactly(count($revision_ids)))
      ->method('createInstance')
      ->withConsecutive(
        ...array_map(function ($revision_id) {
          return ["menu_link_content:{$revision_id}", $this->isType('array')];
        }, $revision_ids)
      )
      ->willReturnOnConsecutiveCalls(...$menu_link_plugins);

    // Mock the container to return our menu link manager.
    $container = $this->createMock('Symfony\Component\DependencyInjection\ContainerInterface');
    $container->method('get')
      ->with('plugin.manager.menu.link')
      ->willReturn($menu_link_manager);
    \Drupal::setContainer($container);

    // Call the method under test.
    $result = $this->menuRevisionManager->getMenuTreeLinksFromMenuRevision($menu_id, $menu_revision_id);

    // Assert the result.
    $this->assertIsArray($result);
    $this->assertCount(count($revision_ids), $result);

    foreach ($revision_ids as $revision_id) {
      $uuid = "uuid-{$revision_id}";
      $this->assertArrayHasKey($uuid, $result);
      $this->assertInstanceOf(MenuLinkInterface::class, $result[$uuid]);
    }
  }

  /**
   * Tests getMenuTreeLinksFromMenuRevision with non-existent menu revision.
   *
   * @covers ::getMenuTreeLinksFromMenuRevision
   */
  public function testGetMenuTreeLinksFromMenuRevisionWithNonExistentRevision() {
    $menu_id = 'test_menu';
    $menu_revision_id = 999;

    // Mock the menu revision query to return empty result.
    $this->entityQuery->method('accessCheck')->willReturnSelf();
    $this->entityQuery->method('condition')->willReturnSelf();
    $this->entityQuery->method('execute')->willReturn([]);

    // Database select should not be called.
    $this->database->expects($this->never())->method('select');

    // Call the method under test.
    $result = $this->menuRevisionManager->getMenuTreeLinksFromMenuRevision($menu_id, $menu_revision_id);

    // Assert empty result.
    $this->assertIsArray($result);
    $this->assertEmpty($result);
  }

  /**
   * Tests getMenuTreeLinksFromMenuRevision with no menu links.
   *
   * @covers ::getMenuTreeLinksFromMenuRevision
   */
  public function testGetMenuTreeLinksFromMenuRevisionWithNoMenuLinks() {
    $menu_id = 'empty_menu';
    $menu_revision_id = 123;

    // Mock the menu revision query.
    $this->entityQuery->method('accessCheck')->willReturnSelf();
    $this->entityQuery->method('condition')->willReturnSelf();
    $this->entityQuery->method('execute')->willReturn([$menu_revision_id]);

    // Mock the database select query to return empty results.
    $this->selectQuery->method('fields')->willReturnSelf();
    $this->selectQuery->method('condition')->willReturnSelf();
    $this->selectQuery->method('execute')->willReturn($this->statement);
    $this->statement->method('fetchCol')->willReturn([]);

    // Menu link storage should not be called.
    $this->menuLinkStorage->expects($this->never())->method('loadRevision');

    // Call the method under test.
    $result = $this->menuRevisionManager->getMenuTreeLinksFromMenuRevision($menu_id, $menu_revision_id);

    // Assert empty result.
    $this->assertIsArray($result);
    $this->assertEmpty($result);
  }

  /**
   * Tests getMenuTreeLinksFromMenuRevision with corrupted menu link revision.
   *
   * @covers ::getMenuTreeLinksFromMenuRevision
   */
  public function testGetMenuTreeLinksFromMenuRevisionWithCorruptedLinks() {
    $menu_id = 'test_menu';
    $menu_revision_id = 123;
    $revision_ids = [10, 20, 30];

    // Mock the menu revision query.
    $this->entityQuery->method('accessCheck')->willReturnSelf();
    $this->entityQuery->method('condition')->willReturnSelf();
    $this->entityQuery->method('execute')->willReturn([$menu_revision_id]);

    // Mock the database select query.
    $this->selectQuery->method('fields')->willReturnSelf();
    $this->selectQuery->method('condition')->willReturnSelf();
    $this->selectQuery->method('execute')->willReturn($this->statement);
    $this->statement->method('fetchCol')->willReturn($revision_ids);

    // Configure loadRevision to return NULL for corrupted links.
    $this->menuLinkStorage->expects($this->exactly(count($revision_ids)))
      ->method('loadRevision')
      ->willReturn(NULL);

    // Call the method under test.
    $result = $this->menuRevisionManager->getMenuTreeLinksFromMenuRevision($menu_id, $menu_revision_id);

    // Assert empty result since all links are corrupted.
    $this->assertIsArray($result);
    $this->assertEmpty($result);
  }

  /**
   * Tests getMenuTreeLinksFromMenuRevision with mixed valid and invalid links.
   *
   * @covers ::getMenuTreeLinksFromMenuRevision
   */
  public function testGetMenuTreeLinksFromMenuRevisionWithMixedLinks() {
    $menu_id = 'test_menu';
    $menu_revision_id = 123;
    $revision_ids = [10, 20, 30];

    // Mock the menu revision query.
    $this->entityQuery->method('accessCheck')->willReturnSelf();
    $this->entityQuery->method('condition')->willReturnSelf();
    $this->entityQuery->method('execute')->willReturn([$menu_revision_id]);

    // Mock the database select query.
    $this->selectQuery->method('fields')->willReturnSelf();
    $this->selectQuery->method('condition')->willReturnSelf();
    $this->selectQuery->method('execute')->willReturn($this->statement);
    $this->statement->method('fetchCol')->willReturn($revision_ids);

    // Create mock for only one valid menu link revision.
    $valid_revision_id = 20;
    $menu_link_revision = $this->createMock(MenuLinkContent::class);
    $uuid = "uuid-{$valid_revision_id}";
    $plugin_id = "menu_link_content:{$valid_revision_id}";

    $menu_link_revision->method('uuid')->willReturn($uuid);
    $menu_link_revision->method('getPluginId')->willReturn($plugin_id);
    $menu_link_revision->method('getPluginDefinition')->willReturn([
      'id' => $plugin_id,
      'title' => "Valid Menu Item",
      'menu_name' => $menu_id,
    ]);

    // Configure loadRevision to return valid link only for revision 20.
    $this->menuLinkStorage->expects($this->exactly(count($revision_ids)))
      ->method('loadRevision')
      ->willReturnCallback(function($revision_id) use ($valid_revision_id, $menu_link_revision) {
        return $revision_id === $valid_revision_id ? $menu_link_revision : NULL;
      });

    // Mock the menu link manager service.
    $menu_link_manager = $this->createMock(MenuLinkManagerInterface::class);
    $menu_link_plugin = $this->createMock(MenuLinkInterface::class);
    $menu_link_plugin->method('getOptions')->willReturn([]);
    $menu_link_plugin->expects($this->once())
      ->method('updateLink')
      ->with(['options' => ['overrideRevisionID' => $valid_revision_id]], FALSE);

    $menu_link_manager->expects($this->once())
      ->method('createInstance')
      ->with($plugin_id, $this->isType('array'))
      ->willReturn($menu_link_plugin);

    // Mock the container.
    $container = $this->createMock('Symfony\Component\DependencyInjection\ContainerInterface');
    $container->method('get')
      ->with('plugin.manager.menu.link')
      ->willReturn($menu_link_manager);
    \Drupal::setContainer($container);

    // Call the method under test.
    $result = $this->menuRevisionManager->getMenuTreeLinksFromMenuRevision($menu_id, $menu_revision_id);

    // Assert result contains only the valid link.
    $this->assertIsArray($result);
    $this->assertCount(1, $result);
    $this->assertArrayHasKey($uuid, $result);
    $this->assertInstanceOf(MenuLinkInterface::class, $result[$uuid]);
  }

  /**
   * Tests getMenuTreeLinksFromMenuRevision with menu link manager exception.
   *
   * @covers ::getMenuTreeLinksFromMenuRevision
   */
  public function testGetMenuTreeLinksFromMenuRevisionWithMenuLinkManagerException() {
    $menu_id = 'test_menu';
    $menu_revision_id = 123;
    $revision_ids = [10];

    // Mock the menu revision query.
    $this->entityQuery->method('accessCheck')->willReturnSelf();
    $this->entityQuery->method('condition')->willReturnSelf();
    $this->entityQuery->method('execute')->willReturn([$menu_revision_id]);

    // Mock the database select query.
    $this->selectQuery->method('fields')->willReturnSelf();
    $this->selectQuery->method('condition')->willReturnSelf();
    $this->selectQuery->method('execute')->willReturn($this->statement);
    $this->statement->method('fetchCol')->willReturn($revision_ids);

    // Create mock menu link revision.
    $menu_link_revision = $this->createMock(MenuLinkContent::class);
    $menu_link_revision->method('uuid')->willReturn('test-uuid');
    $menu_link_revision->method('getPluginId')->willReturn('menu_link_content:10');
    $menu_link_revision->method('getPluginDefinition')->willReturn([]);

    $this->menuLinkStorage->method('loadRevision')->willReturn($menu_link_revision);

    // Mock menu link manager to throw exception.
    $menu_link_manager = $this->createMock(MenuLinkManagerInterface::class);
    $menu_link_manager->method('createInstance')
      ->willThrowException(new \Exception('Plugin creation failed'));

    // Mock the container.
    $container = $this->createMock('Symfony\Component\DependencyInjection\ContainerInterface');
    $container->method('get')
      ->with('plugin.manager.menu.link')
      ->willReturn($menu_link_manager);
    \Drupal::setContainer($container);

    // Expect exception to be thrown.
    $this->expectException(\Exception::class);
    $this->expectExceptionMessage('Plugin creation failed');

    // Call the method under test.
    $this->menuRevisionManager->getMenuTreeLinksFromMenuRevision($menu_id, $menu_revision_id);
  }
}
