<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\Menu\MenuLinkInterface;
use Drupal\Core\Menu\MenuLinkTree;
use Drupal\Core\Menu\MenuLinkTreeElement;
use Drupal\Core\Menu\MenuTreeParameters;
use Drupal\menu_revisions\Services\MenuHierarchyManager;
use Drupal\menu_revisions\Services\MenuRevisionManager;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Services\MenuRevisionManager
 * @group menu_revisions
 */
class MenuRevisionManagerIsMenuRevisionableTest extends UnitTestCase {

  /**
   * The menu revision manager under test.
   *
   * @var \Drupal\menu_revisions\Services\MenuRevisionManager
   */
  protected $menuRevisionManager;

  /**
   * The mocked menu link tree service.
   *
   * @var \Drupal\Core\Menu\MenuLinkTree|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $menuLinkTree;

  /**
   * The mocked entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The mocked current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $currentUser;

  /**
   * The mocked database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $database;

  /**
   * The mocked logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $loggerFactory;

  /**
   * The mocked hierarchy manager.
   *
   * @var \Drupal\menu_revisions\Services\MenuHierarchyManager|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $hierarchyManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mocks for dependencies
    $this->entityTypeManager = $this->createMock('\Drupal\Core\Entity\EntityTypeManagerInterface');
    $this->currentUser = $this->createMock('\Drupal\Core\Session\AccountProxyInterface');
    $this->database = $this->createMock('\Drupal\Core\Database\Connection');
    $this->loggerFactory = $this->createMock('\Drupal\Core\Logger\LoggerChannelFactoryInterface');
    $this->hierarchyManager = $this->createMock(MenuHierarchyManager::class);

    // Mock the logger channel
    $logger = $this->createMock('\Drupal\Core\Logger\LoggerChannelInterface');
    $this->loggerFactory->expects($this->any())
      ->method('get')
      ->with('menu_revision')
      ->willReturn($logger);

    // Create a mock for the MenuLinkTree service
    $this->menuLinkTree = $this->createMock(MenuLinkTree::class);

    // Register the MenuLinkTree service in the Drupal service container
    $container = new \Drupal\Core\DependencyInjection\ContainerBuilder();
    $container->set('menu.link_tree', $this->menuLinkTree);
    \Drupal::setContainer($container);

    // Create the class under test
    $this->menuRevisionManager = new MenuRevisionManager(
      $this->entityTypeManager,
      $this->currentUser,
      $this->database,
      $this->loggerFactory,
      $this->hierarchyManager
    );
  }

  /**
   * Data provider for testIsMenuRevisionable.
   *
   * @return array
   *   Test data.
   */
  public function isMenuRevisionableDataProvider() {
    return [
      'all menu_link_content links' => [
        'links' => [
          $this->createMenuTreeElement('menu_link_content:uuid1'),
          $this->createMenuTreeElement('menu_link_content:uuid2'),
          $this->createMenuTreeElement('menu_link_content:uuid3'),
        ],
        'expected' => TRUE,
      ],
      'mixed links' => [
        'links' => [
          $this->createMenuTreeElement('menu_link_content:uuid1'),
          $this->createMenuTreeElement('system.admin_content'),
          $this->createMenuTreeElement('menu_link_content:uuid3'),
        ],
        'expected' => FALSE,
      ],
      'all system links' => [
        'links' => [
          $this->createMenuTreeElement('system.admin_content'),
          $this->createMenuTreeElement('system.admin_structure'),
        ],
        'expected' => FALSE,
      ],
      'empty menu' => [
        'links' => [],
        'expected' => TRUE,
      ],
    ];
  }

  /**
   * Helper method to create a MenuTreeElement with a specific plugin ID.
   *
   * @param string $plugin_id
   *   The plugin ID for the menu link.
   *
   * @return \Drupal\Core\Menu\MenuLinkTreeElement
   *   A menu tree element mock with the given plugin ID.
   */
  protected function createMenuTreeElement($plugin_id) {
    $menu_link = $this->createMock(MenuLinkInterface::class);
    $menu_link->expects($this->any())
      ->method('getPluginId')
      ->willReturn($plugin_id);

    return new MenuLinkTreeElement($menu_link, FALSE, 0, FALSE, []);
  }

  /**
   * Tests the isMenuRevisionable method.
   *
   * @param array $links
   *   The menu links to test with.
   * @param bool $expected
   *   The expected result of isMenuRevisionable.
   *
   * @covers ::isMenuRevisionable
   * @dataProvider isMenuRevisionableDataProvider
   */
  public function testIsMenuRevisionable(array $links, bool $expected) {
    $menu_id = 'test_menu';

    // Setup the mock to return our test links
    $this->menuLinkTree->expects($this->once())
      ->method('load')
      ->with($menu_id, $this->isInstanceOf(MenuTreeParameters::class))
      ->willReturn($links);

    $result = $this->menuRevisionManager->isMenuRevisionable($menu_id);
    $this->assertEquals($expected, $result);
  }

  /**
   * Tests isMenuRevisionable with a menu that throws an exception when loading.
   *
   * @covers ::isMenuRevisionable
   */
  public function testIsMenuRevisionableWithException() {
    $menu_id = 'test_menu';

    // Setup the mock to throw an exception
    $this->menuLinkTree->expects($this->once())
      ->method('load')
      ->with($menu_id, $this->isInstanceOf(MenuTreeParameters::class))
      ->willThrowException(new \Exception('Test exception'));

    // We expect an exception to be thrown
    $this->expectException(\Exception::class);
    $this->expectExceptionMessage('Test exception');

    $this->menuRevisionManager->isMenuRevisionable($menu_id);
  }

}
