<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\StatementInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\menu_revisions\Services\MenuHierarchyManager;
use Drupal\menu_revisions\Services\MenuRevisionManager;
use Drupal\Tests\UnitTestCase;
use Prophecy\PhpUnit\ProphecyTrait;
use Prophecy\Argument;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Services\MenuRevisionManager
 * @group menu_revisions
 */
class MenuRevisionManagerRevertMenuToRevisionTest extends UnitTestCase {

  use ProphecyTrait;

  /**
   * The menu revision manager service under test.
   *
   * @var \Drupal\menu_revisions\Services\MenuRevisionManager
   */
  protected $menuRevisionManager;

  /**
   * The mocked database connection.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $database;

  /**
   * The mocked entity type manager.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $entityTypeManager;

  /**
   * The mocked current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $currentUser;

  /**
   * The mocked logger factory (PHPUnit mock).
   *
   * @var \PHPUnit\Framework\MockObject\MockObject
   */
  protected $loggerFactory;

  /**
   * The mocked logger channel (PHPUnit mock).
   *
   * @var \PHPUnit\Framework\MockObject\MockObject
   */
  protected $loggerChannel;

  /**
   * The mocked hierarchy manager.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $hierarchyManager;

  /**
   * The mocked menu link storage.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $menuLinkStorage;

  /**
   * The mocked menu revision storage.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $menuRevisionStorage;

  /**
   * The mocked container.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $container;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Prophecy mocks
    $this->database = $this->prophesize(Connection::class);
    $this->entityTypeManager = $this->prophesize(EntityTypeManagerInterface::class);
    $this->currentUser = $this->createMock(AccountProxyInterface::class);

    $this->hierarchyManager = $this->prophesize(MenuHierarchyManager::class);
    $this->menuLinkStorage = $this->prophesize(EntityStorageInterface::class);
    $this->menuRevisionStorage = $this->prophesize(EntityStorageInterface::class);

    // Entity type manager returns storages
    $this->entityTypeManager->getStorage('menu_link_content')
      ->willReturn($this->menuLinkStorage->reveal());
    $this->entityTypeManager->getStorage('menu_revision')
      ->willReturn($this->menuRevisionStorage->reveal());

    // Use PHPUnit mocks for logger factory/channel to avoid Prophecy void-method quirks.
    $this->loggerChannel = $this->createMock(LoggerChannelInterface::class);
    $this->loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);

    // Return channel for any logger name (safer than constraining to a specific string).
    $this->loggerFactory
      ->method('get')
      ->willReturn($this->loggerChannel);

    // Container and static services (Prophecy)
    $this->container = $this->prophesize(ContainerInterface::class);

    $menuLinkPluginManager = $this->prophesize('\Drupal\Core\Menu\MenuLinkManagerInterface');
    $menuLinkPluginManager->rebuild()->willReturn(NULL);

    $routerBuilder = $this->prophesize('\Drupal\Core\Routing\RouteBuilderInterface');
    $routerBuilder->rebuild()->willReturn(NULL);

    $menuCache = $this->prophesize('\Drupal\Core\Cache\CacheBackendInterface');
    $menuCache->deleteAll()->willReturn(NULL);

    $renderCache = $this->prophesize('\Drupal\Core\Cache\CacheBackendInterface');
    $renderCache->invalidateAll()->willReturn(NULL);

    // Cache tags invalidator (Cache::invalidateTags uses 'cache_tags.invalidator' service).
    $cacheTagsInvalidator = $this->prophesize('\Drupal\Core\Cache\CacheTagsInvalidatorInterface');
    $cacheTagsInvalidator->invalidateTags(Argument::any())->will(function () {});

    $this->container->get('plugin.manager.menu.link')->willReturn($menuLinkPluginManager->reveal());
    $this->container->get('router.builder')->willReturn($routerBuilder->reveal());
    $this->container->get('cache.menu')->willReturn($menuCache->reveal());
    $this->container->get('cache.render')->willReturn($renderCache->reveal());
    $this->container->get('cache_tags.invalidator')->willReturn($cacheTagsInvalidator->reveal());

    // Set the global container so \Drupal::service() works in the tested method.
    \Drupal::setContainer($this->container->reveal());

    // Instantiate the service under test with the correct constructor order:
    // (EntityTypeManagerInterface, AccountProxyInterface, Connection, LoggerChannelFactoryInterface, MenuHierarchyManager)
    $this->menuRevisionManager = new MenuRevisionManager(
      $this->entityTypeManager->reveal(),
      $this->currentUser,
      $this->database->reveal(),
      $this->loggerFactory,
      $this->hierarchyManager->reveal()
    );
  }

  /**
   * Tests reverting a menu when the revision does not exist.
   *
   * @covers ::revertMenuToRevision
   */
  public function testRevertMenuToRevisionWhenRevisionDoesNotExist() {
    $menu_revision_id = 123;

    // Transaction stub - no assertion required.
    $transaction = new class {
      public function rollBack() {}
    };
    $this->database->startTransaction('menu_revision_revert')
      ->willReturn($transaction);

    // menu_revision storage returns NULL
    $this->menuRevisionStorage->load($menu_revision_id)
      ->willReturn(NULL);

    $result = $this->menuRevisionManager->revertMenuToRevision($menu_revision_id);

    $this->assertFalse($result);
  }

  /**
   * Tests reverting a menu when there's insufficient revision data.
   *
   * @covers ::revertMenuToRevision
   */
  public function testRevertMenuToRevisionWithInsufficientData() {
    $menu_revision_id = 123;
    $menu_name = 'main';

    // Transaction stub.
    $transaction = new class {
      public function rollBack() {}
    };
    $this->database->startTransaction('menu_revision_revert')
      ->willReturn($transaction);

    // Mock revision entity
    $menu_revision = $this->prophesize('\Drupal\menu_revisions\Entity\MenuRevision');
    $menu_revision->getMenuName()->willReturn($menu_name);

    $this->menuRevisionStorage->load($menu_revision_id)
      ->willReturn($menu_revision->reveal());

    // Mock select chain that returns no revision links.
    $select = $this->prophesize('\Drupal\Core\Database\Query\Select');
    $select->fields('mrl')->willReturn($select->reveal());
    $select->condition('menu_revision_id', $menu_revision_id)->willReturn($select->reveal());

    $statement = $this->prophesize(StatementInterface::class);
    $statement->fetchAllAssoc('menu_link_content_id')->willReturn([]);

    $select->execute()->willReturn($statement->reveal());
    $this->database->select('menu_revision_link', 'mrl')->willReturn($select->reveal());

    // Hierarchy returns empty
    $this->hierarchyManager->getMenuHierarchy($menu_revision_id)
      ->willReturn([]);

    $result = $this->menuRevisionManager->revertMenuToRevision($menu_revision_id);

    $this->assertFalse($result);
  }

  /**
   * Tests a successful menu revert operation with existing and new links.
   *
   * @covers ::revertMenuToRevision
   */
  public function testSuccessfulRevertMenuToRevision() {
    $menu_revision_id = 123;
    $menu_name = 'main';

    // Transaction stub.
    $transaction = new class {
      public function rollBack() {}
    };
    $this->database->startTransaction('menu_revision_revert')
      ->willReturn($transaction);

    // Mock menu revision entity
    $menu_revision = $this->prophesize('\Drupal\menu_revisions\Entity\MenuRevision');
    $menu_revision->getMenuName()->willReturn($menu_name);
    $this->menuRevisionStorage->load($menu_revision_id)
      ->willReturn($menu_revision->reveal());

    // Mock revision links returned by DB
    $revision_link1 = (object) [
      'menu_link_content_id' => 101,
      'menu_revision_id' => $menu_revision_id,
      'menu_link_revision_id' => 201,
    ];
    $revision_link2 = (object) [
      'menu_link_content_id' => 102,
      'menu_revision_id' => $menu_revision_id,
      'menu_link_revision_id' => 202,
    ];
    $revision_links = [
      101 => $revision_link1,
      102 => $revision_link2,
    ];

    $select = $this->prophesize('\Drupal\Core\Database\Query\Select');
    $select->fields('mrl')->willReturn($select->reveal());
    $select->condition('menu_revision_id', $menu_revision_id)->willReturn($select->reveal());

    $statement = $this->prophesize(StatementInterface::class);
    $statement->fetchAllAssoc('menu_link_content_id')->willReturn($revision_links);

    $select->execute()->willReturn($statement->reveal());
    $this->database->select('menu_revision_link', 'mrl')->willReturn($select->reveal());

    // Mock hierarchy mapping
    $hierarchy_data = [
      'uuid-1' => ['id' => 101, 'weight' => 0, 'parent' => ''],
      'uuid-2' => ['id' => 102, 'weight' => 1, 'parent' => 'menu_link_content:uuid-1'],
    ];
    $this->hierarchyManager->getMenuHierarchy($menu_revision_id)
      ->willReturn($hierarchy_data);

    // Mock entity query for current links
    $entityQuery = $this->prophesize('\Drupal\Core\Entity\Query\QueryInterface');
    $entityQuery->accessCheck(TRUE)->willReturn($entityQuery->reveal());
    $entityQuery->condition('menu_name', $menu_name)->willReturn($entityQuery->reveal());
    $entityQuery->execute()->willReturn([101]);

    $this->menuLinkStorage->getQuery()->willReturn($entityQuery->reveal());

    // Mock an existing MenuLinkContent entity (loaded via loadMultiple)
    $existingLink = $this->prophesize(MenuLinkContent::class);
    $existingLink->id()->willReturn(101);
    $existingLink->uuid()->willReturn('uuid-1');

    // Allow getFieldDefinitions() — service may call it internally.
    $existingLink->getFieldDefinitions()->willReturn([]);

    // Fields structure returned by loadRevision/getFields
    $field = $this->prophesize('\Drupal\Core\Field\FieldItemListInterface');
    $field->getValue()->willReturn([['value' => 'v']]);

    $fields = [
      'title' => $field->reveal(),
      'link' => $field->reveal(),
      'description' => $field->reveal(),
      'menu_name' => $field->reveal(),
      'expanded' => $field->reveal(),
    ];

    $existingLink->getFields(FALSE)->willReturn($fields);
    $existingLink->set(Argument::any(), Argument::any())->willReturn($existingLink->reveal());
    $existingLink->setNewRevision(TRUE)->willReturn(NULL);
    $existingLink->save()->willReturn(101);

    $this->menuLinkStorage->loadMultiple([101])->willReturn([101 => $existingLink->reveal()]);

    // Mock menu_link_content revisions returned by loadRevision()
    $menuLinkRevision1 = $this->prophesize(MenuLinkContent::class);
    $menuLinkRevision1->getFields(FALSE)->willReturn($fields);
    $menuLinkRevision1->getFieldDefinitions()->willReturn([]);

    $menuLinkRevision2 = $this->prophesize(MenuLinkContent::class);
    $menuLinkRevision2->getFields(FALSE)->willReturn($fields);
    $menuLinkRevision2->getFieldDefinitions()->willReturn([]);

    $this->menuLinkStorage->loadRevision(201)->willReturn($menuLinkRevision1->reveal());
    $this->menuLinkStorage->loadRevision(202)->willReturn($menuLinkRevision2->reveal());

    // Mock create() for new link (match any array argument)
    $newLink = $this->prophesize(MenuLinkContent::class);
    $newLink->id()->willReturn(102);
    $newLink->set(Argument::any(), Argument::any())->willReturn($newLink->reveal());
    $newLink->enforceIsNew()->willReturn(NULL);
    $newLink->save()->willReturn(102);
    $newLink->getFieldDefinitions()->willReturn([]);

    $this->menuLinkStorage->create(Argument::any())->willReturn($newLink->reveal());

    // Partial mock of the manager to stub out the private finalization methods.
    $menuRevisionManagerMock = $this->getMockBuilder(MenuRevisionManager::class)
      ->setConstructorArgs([
        $this->entityTypeManager->reveal(),
        $this->currentUser,
        $this->database->reveal(),
        $this->loggerFactory,
        $this->hierarchyManager->reveal(),
      ])
      ->onlyMethods(['createRevisionFromMenu', 'cleanDefaultStatusForItemsNotInMenuRevisionID'])
      ->getMock();

    $menuRevisionManagerMock->expects($this->once())
      ->method('createRevisionFromMenu')
      ->with($menu_name)
      ->willReturn(124);

    $menuRevisionManagerMock->expects($this->once())
      ->method('cleanDefaultStatusForItemsNotInMenuRevisionID')
      ->with(124);

    $result = $menuRevisionManagerMock->revertMenuToRevision($menu_revision_id);

    $this->assertEquals(124, $result);
  }

  /**
   * Tests error handling during the revert process.
   *
   * @covers ::revertMenuToRevision
   */
  public function testErrorHandlingDuringRevert() {
    $menu_revision_id = 123;
    $menu_name = 'main';

    // Create a PHPUnit mock for transaction so we can assert rollBack() is called.
    $transaction = $this->getMockBuilder(\stdClass::class)
      ->addMethods(['rollBack'])
      ->getMock();
    $transaction->expects($this->once())->method('rollBack');

    $this->database->startTransaction('menu_revision_revert')
      ->willReturn($transaction);

    // Mock revision entity
    $menu_revision = $this->prophesize('\Drupal\menu_revisions\Entity\MenuRevision');
    $menu_revision->getMenuName()->willReturn($menu_name);
    $this->menuRevisionStorage->load($menu_revision_id)
      ->willReturn($menu_revision->reveal());

    // Make the select throw an exception when condition() is called to simulate DB error.
    $select = $this->prophesize('\Drupal\Core\Database\Query\Select');
    $select->fields('mrl')->willReturn($select->reveal());
    $select->condition('menu_revision_id', $menu_revision_id)
      ->willThrow(new \Exception('Database error'));

    $this->database->select('menu_revision_link', 'mrl')->willReturn($select->reveal());

    $result = $this->menuRevisionManager->revertMenuToRevision($menu_revision_id);

    $this->assertFalse($result);
  }

  /**
   * Tests the edge case of a partial hierarchy in the revision.
   *
   * @covers ::revertMenuToRevision
   */
  public function testPartialHierarchyInRevision() {
    $menu_revision_id = 123;
    $menu_name = 'main';

    // Transaction stub.
    $transaction = new class {
      public function rollBack() {}
    };
    $this->database->startTransaction('menu_revision_revert')
      ->willReturn($transaction);

    // Mock revision entity
    $menu_revision = $this->prophesize('\Drupal\menu_revisions\Entity\MenuRevision');
    $menu_revision->getMenuName()->willReturn($menu_name);
    $this->menuRevisionStorage->load($menu_revision_id)
      ->willReturn($menu_revision->reveal());

    // Revision links - only 101 exists
    $revision_link1 = (object) [
      'menu_link_content_id' => 101,
      'menu_revision_id' => $menu_revision_id,
      'menu_link_revision_id' => 201,
    ];
    $revision_links = [101 => $revision_link1];

    $select = $this->prophesize('\Drupal\Core\Database\Query\Select');
    $select->fields('mrl')->willReturn($select->reveal());
    $select->condition('menu_revision_id', $menu_revision_id)->willReturn($select->reveal());

    $statement = $this->prophesize(StatementInterface::class);
    $statement->fetchAllAssoc('menu_link_content_id')->willReturn($revision_links);

    $select->execute()->willReturn($statement->reveal());
    $this->database->select('menu_revision_link', 'mrl')->willReturn($select->reveal());

    // Hierarchy contains an extra item (102) not present in revision_links
    $hierarchy_data = [
      'uuid-1' => ['id' => 101, 'weight' => 0, 'parent' => ''],
      'uuid-2' => ['id' => 102, 'weight' => 1, 'parent' => 'menu_link_content:uuid-1'],
    ];
    $this->hierarchyManager->getMenuHierarchy($menu_revision_id)
      ->willReturn($hierarchy_data);

    // Current links query returns only 101
    $entityQuery = $this->prophesize('\Drupal\Core\Entity\Query\QueryInterface');
    $entityQuery->accessCheck(TRUE)->willReturn($entityQuery->reveal());
    $entityQuery->condition('menu_name', $menu_name)->willReturn($entityQuery->reveal());
    $entityQuery->execute()->willReturn([101]);

    $this->menuLinkStorage->getQuery()->willReturn($entityQuery->reveal());

    // Existing entity for 101
    $existingLink = $this->prophesize(MenuLinkContent::class);
    $existingLink->id()->willReturn(101);
    $existingLink->uuid()->willReturn('uuid-1');
    $existingLink->getFieldDefinitions()->willReturn([]);

    $field = $this->prophesize('\Drupal\Core\Field\FieldItemListInterface');
    $field->getValue()->willReturn([['value' => 'v']]);

    $fields = [
      'title' => $field->reveal(),
      'link' => $field->reveal(),
      'description' => $field->reveal(),
      'menu_name' => $field->reveal(),
      'expanded' => $field->reveal(),
    ];

    $existingLink->getFields(FALSE)->willReturn($fields);
    $existingLink->set(Argument::any(), Argument::any())->willReturn($existingLink->reveal());
    $existingLink->setNewRevision(TRUE)->willReturn(NULL);
    $existingLink->save()->willReturn(101);

    $this->menuLinkStorage->loadMultiple([101])->willReturn([101 => $existingLink->reveal()]);

    // loadRevision for 201 exists
    $menuLinkRevision1 = $this->prophesize(MenuLinkContent::class);
    $menuLinkRevision1->getFields(FALSE)->willReturn($fields);
    $menuLinkRevision1->getFieldDefinitions()->willReturn([]);
    $this->menuLinkStorage->loadRevision(201)->willReturn($menuLinkRevision1->reveal());

    // Partial mock manager to avoid finalization internals
    $menuRevisionManagerMock = $this->getMockBuilder(MenuRevisionManager::class)
      ->setConstructorArgs([
        $this->entityTypeManager->reveal(),
        $this->currentUser,
        $this->database->reveal(),
        $this->loggerFactory,
        $this->hierarchyManager->reveal(),
      ])
      ->onlyMethods(['createRevisionFromMenu', 'cleanDefaultStatusForItemsNotInMenuRevisionID'])
      ->getMock();

    $menuRevisionManagerMock->expects($this->once())
      ->method('createRevisionFromMenu')
      ->with($menu_name)
      ->willReturn(124);

    $menuRevisionManagerMock->expects($this->once())
      ->method('cleanDefaultStatusForItemsNotInMenuRevisionID')
      ->with(124);

    $result = $menuRevisionManagerMock->revertMenuToRevision($menu_revision_id);

    $this->assertEquals(124, $result);
  }

}
