<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Tests\UnitTestCase;

// Load the .module file for testing procedural functions.
require_once __DIR__ . '/../../../menu_revisions.module';

/**
 * Tests for menu_revisions.module procedural functions.
 *
 * @group menu_revisions
 */
class MenuRevisionsModuleTest extends UnitTestCase
{
  protected function setUp(): void
  {
    parent::setUp();

    // Setup minimal container for procedural function tests.
    $container = new ContainerBuilder();
    $routeMatch = $this->createMock(RouteMatchInterface::class);
    $container->set('current_route_match', $routeMatch);
    \Drupal::setContainer($container);
  }

  /**
   * Tests menu_revisions_disableExternalMenuImport().
   */
  public function testDisableExternalMenuImportOnPreviewRoute(): void
  {
    $routeMatch = $this->createMock(RouteMatchInterface::class);
    $routeMatch->method('getRouteName')->willReturn('menu_revisions.revision_preview');

    $container = new ContainerBuilder();
    $container->set('current_route_match', $routeMatch);
    \Drupal::setContainer($container);

    $variables = [
      'use_external_menu' => true,
      'global_menu' => ['some' => 'data'],
    ];

    menu_revisions_disableExternalMenuImport($variables);

    $this->assertFalse($variables['use_external_menu']);
    $this->assertArrayNotHasKey('global_menu', $variables);
  }

  /**
   * Tests menu_revisions_disableExternalMenuImport() does nothing on other routes.
   */
  public function testDisableExternalMenuImportOnOtherRoute(): void
  {
    $routeMatch = $this->createMock(RouteMatchInterface::class);
    $routeMatch->method('getRouteName')->willReturn('some.other.route');

    $container = new ContainerBuilder();
    $container->set('current_route_match', $routeMatch);
    \Drupal::setContainer($container);

    $variables = [
      'use_external_menu' => true,
      'global_menu' => ['some' => 'data'],
    ];
    $originalVariables = $variables;

    menu_revisions_disableExternalMenuImport($variables);

    $this->assertEquals($originalVariables, $variables);
  }

  /**
   * Tests _menu_revisions_get_latest_draft_revision().
   */
  public function testGetLatestDraftRevisionReturnsEntity(): void
  {
    $revisionEntity = $this->createMock(\Drupal\Core\Entity\EntityInterface::class);
    $storage = $this->createMock(EntityStorageInterface::class);
    $query = $this->createMock(QueryInterface::class);

    $storage->method('getQuery')->willReturn($query);
    $query->method('accessCheck')->with(true)->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('sort')->willReturnSelf();
    $query->method('range')->willReturnSelf();
    $query->method('execute')->willReturn([42]);
    $storage->method('load')->with(42)->willReturn($revisionEntity);

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')->with('menu_revision')->willReturn($storage);

    $container = new ContainerBuilder();
    $container->set('entity_type.manager', $entityTypeManager);
    \Drupal::setContainer($container);

    $result = _menu_revisions_get_latest_draft_revision('main');

    $this->assertSame($revisionEntity, $result);
  }

  /**
   * Tests _menu_revisions_get_latest_draft_revision() returns NULL when none found.
   */
  public function testGetLatestDraftRevisionReturnsNullWhenNotFound(): void
  {
    $storage = $this->createMock(EntityStorageInterface::class);
    $query = $this->createMock(QueryInterface::class);

    $storage->method('getQuery')->willReturn($query);
    $query->method('accessCheck')->with(true)->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('sort')->willReturnSelf();
    $query->method('range')->willReturnSelf();
    $query->method('execute')->willReturn([]);

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')->with('menu_revision')->willReturn($storage);

    $container = new ContainerBuilder();
    $container->set('entity_type.manager', $entityTypeManager);
    \Drupal::setContainer($container);

    $result = _menu_revisions_get_latest_draft_revision('main');

    $this->assertNull($result);
  }
}

