<?php

namespace Drupal\Tests\menu_revisions\Unit\Utils;

use Drupal\Core\Menu\MenuLinkInterface;
use Drupal\Core\Menu\MenuLinkTreeElement;
use Drupal\menu_revisions\Utils\MenuTreeUtil;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Utils\MenuTreeUtil
 * @group menu_revisions
 */
class MenuTreeUtilTest extends UnitTestCase
{
  /**
   * @covers ::buildMenuTreeWithSubTree
   */
  public function testBuildMenuTreeWithSubTreeBuildsHierarchyAndRemovesChildren(): void
  {
    // Create mock menu link objects.
    $linkA = $this->createMock(MenuLinkInterface::class);
    $linkA->method('updateLink');
    $linkB = $this->createMock(MenuLinkInterface::class);
    $linkB->method('updateLink');
    $linkC = $this->createMock(MenuLinkInterface::class);
    $linkC->method('updateLink');

    // Build flat tree with parent and children.
    $elementA = new MenuLinkTreeElement($linkA, false, 1, false, []);
    $elementB = new MenuLinkTreeElement($linkB, false, 1, false, []);
    $elementC = new MenuLinkTreeElement($linkC, false, 1, false, []);

    $menuTreeFlat = [
      'uuid-a' => $elementA,
      'uuid-b' => $elementB,
      'uuid-c' => $elementC,
    ];

    // Hierarchy: A is top-level, B and C are children of A.
    $hierarchy = [
      'menu_link_content:uuid-a' => ['plugin_id' => 'menu_link_content:uuid-a', 'parent' => '', 'weight' => 0],
      'menu_link_content:uuid-b' => ['plugin_id' => 'menu_link_content:uuid-b', 'parent' => 'menu_link_content:uuid-a', 'weight' => 1],
      'menu_link_content:uuid-c' => ['plugin_id' => 'menu_link_content:uuid-c', 'parent' => 'menu_link_content:uuid-a', 'weight' => 2],
    ];

    $util = new MenuTreeUtil();
    $util->buildMenuTreeWithSubTree($menuTreeFlat, $hierarchy);

    // After processing, only top-level item A should remain in the flat array.
    $this->assertArrayHasKey('uuid-a', $menuTreeFlat);
    $this->assertArrayNotHasKey('uuid-b', $menuTreeFlat);
    $this->assertArrayNotHasKey('uuid-c', $menuTreeFlat);

    // A should have 2 children in subtree.
    $this->assertCount(2, $menuTreeFlat['uuid-a']->subtree);
    $this->assertTrue($menuTreeFlat['uuid-a']->hasChildren);
  }

  /**
   * @covers ::buildMenuTreeWithSubTree
   */
  public function testBuildMenuTreeWithSubTreeUpdatesWeights(): void
  {
    $link = $this->createMock(MenuLinkInterface::class);
    $link->expects($this->once())->method('updateLink')->with(['weight' => 5], false);

    $element = new MenuLinkTreeElement($link, false, 1, false, []);
    $menuTreeFlat = ['uuid-x' => $element];
    $hierarchy = [
      'menu_link_content:uuid-x' => ['plugin_id' => 'menu_link_content:uuid-x', 'parent' => '', 'weight' => 5],
    ];

    $util = new MenuTreeUtil();
    $util->buildMenuTreeWithSubTree($menuTreeFlat, $hierarchy);

    $this->assertArrayHasKey('uuid-x', $menuTreeFlat);
  }

  /**
   * @covers ::buildMenuTreeWithSubTree
   */
  public function testBuildMenuTreeWithSubTreeHandlesEmptyTree(): void
  {
    $menuTreeFlat = [];
    $hierarchy = [];

    $util = new MenuTreeUtil();
    $util->buildMenuTreeWithSubTree($menuTreeFlat, $hierarchy);

    $this->assertEmpty($menuTreeFlat);
  }

  /**
   * @covers ::buildMenuTreeWithSubTree
   */
  public function testBuildMenuTreeWithSubTreeHandlesFlatTreeWithNoChildren(): void
  {
    $linkA = $this->createMock(MenuLinkInterface::class);
    $linkA->method('updateLink');
    $linkB = $this->createMock(MenuLinkInterface::class);
    $linkB->method('updateLink');

    $elementA = new MenuLinkTreeElement($linkA, false, 1, false, []);
    $elementB = new MenuLinkTreeElement($linkB, false, 1, false, []);

    $menuTreeFlat = [
      'uuid-a' => $elementA,
      'uuid-b' => $elementB,
    ];

    // Both are top-level (no parent relationships).
    $hierarchy = [
      'menu_link_content:uuid-a' => ['plugin_id' => 'menu_link_content:uuid-a', 'parent' => '', 'weight' => 0],
      'menu_link_content:uuid-b' => ['plugin_id' => 'menu_link_content:uuid-b', 'parent' => '', 'weight' => 1],
    ];

    $util = new MenuTreeUtil();
    $util->buildMenuTreeWithSubTree($menuTreeFlat, $hierarchy);

    // Both should still be present as top-level items.
    $this->assertArrayHasKey('uuid-a', $menuTreeFlat);
    $this->assertArrayHasKey('uuid-b', $menuTreeFlat);
    $this->assertCount(0, $menuTreeFlat['uuid-a']->subtree);
    $this->assertCount(0, $menuTreeFlat['uuid-b']->subtree);
  }
}

