<?php

namespace Drupal\entities_feed\Form;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\entities_feed\Batch\EntitiesFeedImportBatch;
use Drupal\Core\Extension\ModuleExtensionList;

/**
 * Provides a form for importing entities and fields from CSV.
 */
class EntitiesFeedImportForm extends FormBase {

   /**
    * File system service.
    *
    * @var \Drupal\Core\File\FileSystemInterface
    */
  protected $fileSystem;

  /**
   * Logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

   /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * El servicio para obtener paths de módulos.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected $extensionList;
  
  /**
   * Constructs the EntitiesFeedImporter service.
   */
  public function __construct(FileSystemInterface $file_system, LoggerInterface $logger, MessengerInterface $messenger, EntityTypeManagerInterface $entity_type_manager, ModuleExtensionList $extension_list) {
    $this->fileSystem = $file_system;
    $this->logger = $logger;
    $this->messenger = $messenger;
    $this->entityTypeManager = $entity_type_manager;
    $this->extensionList = $extension_list;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
        $container->get('file_system'),
        $container->get('logger.factory')->get('entities_feed'),
        $container->get('messenger'),
        $container->get('entity_type.manager'),
        $container->get('extension.list.module'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entities_feed_import_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form_state->disableCache();
    // Build tabbed interface.
    $module_name = 'entities_feed';
    $module_path = $this->extensionList->getPath($module_name);  
    $form['#attributes'] = ['enctype' => 'multipart/form-data'];
    $form['#tree'] = TRUE;
    $form['#attached']['library'][] = 'core/drupal.tabs';
    $form['#prefix'] = '<div class="entities-feed-tabs">';
    $form['#suffix'] = '</div>';

    // URLs for CSV templates.
    $entity_template_url = Url::fromUri('base:' . $module_path . '/templates/examples/Entity_feed_Template.xlsx')->toString();
    $fields_template_url = Url::fromUri('base:' . $module_path . 'templates/examples/Entity_feed_Template.xlsx')->toString();

    // Tabs.
    $form['tabs'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Entities Feed Importer export csv from single sheet in layout sheets'),
    ];

    // === Entities tab ===
    $form['entities_tab'] = [
      '#type' => 'details',
      '#title' => $this->t('Carga de entidades'),
      '#group' => 'tabs',
    ];

    $form['entities_tab']['entities_description'] = [
      '#markup' => $this->t('<p>Use this form to import content types or taxonomies from a CSV file.</p>'),
    ];

    $form['entities_tab']['template_link'] = [
      '#type' => 'link',
      '#title' => $this->t('Download entities CSV template'),
      '#url' => Url::fromUri('base:' . $entity_template_url),
      '#attributes' => ['target' => '_blank', 'class' => ['button', 'button--small']],
    ];

    $form['entities_tab']['csv_file'] = [
        '#type' => 'managed_file',
        '#title' => $this->t('Upload CSV file'),
        '#description' => $this->t('Select the CSV file that contains the entities to import.'),
        '#upload_location' => 'public://content/csv_files/',
        '#upload_validators' => [
            'FileExtension' => [
                'extensions' => 'csv',  
            ],
        ],
        '#accept' => '.csv',  
        '#multiple' => FALSE,
    ];

    $form['entities_tab']['delimiter'] = [
      '#type' => 'select',
      '#title' => $this->t('CSV Delimiter'),
      '#options' => [
        ',' => $this->t('Comma (,)'),
        ';' => $this->t('Semicolon (;)'),
        '|' => $this->t('Pipe (|)'),
        '\\t' => $this->t('Tab (\\t)'),
      ],
      '#default_value' => ',',
    ];

    $form['entities_tab']['submit_entities'] = [
      '#type' => 'submit',
      '#value' => $this->t('Importar entidades'),
      '#submit' => ['::submitEntities'],
    ];

    // === Fields tab ===
    $form['fields_tab'] = [
      '#type' => 'details',
      '#title' => $this->t('Carga de campos'),
      '#group' => 'tabs',
    ];

    $form['fields_tab']['fields_description'] = [
      '#markup' => $this->t('<p>Use this form to import fields for existing content types from a CSV file.</p>'),
    ];

    $form['fields_tab']['template_link'] = [
      '#type' => 'link',
      '#title' => $this->t('Download fields CSV template'),
      '#url' => Url::fromUri('base:' . $fields_template_url),
      '#attributes' => ['target' => '_blank', 'class' => ['button', 'button--small']],
    ];

    $form['fields_tab']['csv_file'] = [
        '#type' => 'managed_file',
        '#title' => $this->t('Upload CSV file'),
        '#description' => $this->t('Select the CSV file that contains the fields to import.'),
        '#upload_location' => 'public://content/csv_files/',
        '#upload_validators' => [
          'FileExtension' => [
            'extensions' => 'csv',  
          ],
        ],
        '#accept' => '.csv',  
        '#multiple' => FALSE,
    ];

    $form['fields_tab']['delimiter'] = [
      '#type' => 'select',
      '#title' => $this->t('CSV Delimiter'),
      '#options' => [
        ',' => $this->t('Comma (,)'),
        ';' => $this->t('Semicolon (;)'),
        '|' => $this->t('Pipe (|)'),
        '\\t' => $this->t('Tab (\\t)'),
      ],
      '#default_value' => ',',
    ];

    $form['fields_tab']['submit_fields'] = [
      '#type' => 'submit',
      '#value' => $this->t('Importar campos'),
      '#submit' => ['::submitFields'],
    ];

    return $form;
  }

  /**
   * Submit handler for entities tab.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   * Submit handler for entities tab.
   */
  public function submitEntities(array &$form, FormStateInterface $form_state) {
    $delimiter = $form_state->getValue(['entities_tab', 'delimiter']);
    $fid = $form_state->getValue(['entities_tab', 'csv_file']);
    $response = $this->handleSubmition($fid, $delimiter, 'entities');
    $form_state->setRebuild(FALSE);
    $this->messenger()->addStatus($this->t('Entities imported successfully using delimiter: @delimiter', ['@delimiter' => $delimiter]));
  }

  /**
   * Submit handler for fields tab.
   */
  public function submitFields(array &$form, FormStateInterface $form_state) {
    $delimiter = $form_state->getValue(['fields_tab', 'delimiter']);
    $fid = $form_state->getValue(['fields_tab', 'csv_file']);
    $response = $this->handleSubmition($fid, $delimiter, 'fields');
    $form_state->setRebuild(FALSE);
    $this->messenger()->addStatus($this->t('Fields imported successfully using delimiter: @delimiter', ['@delimiter' => $delimiter]));
  }

  /**
   * Handle submitions
   * @param $fid 
   * Load csv document
   * @param $delimiter
   * Document delimiter
   * @param $process 
   * Load process (Entity or field)
   */
  public function handleSubmition($fid, $delimiter, $process) {
    if ($delimiter === '\t') {
        $delimiter = "\t";
    }
    $destination = $this->entityTypeManager->getStorage('file')->load($fid[0]);
    $uri_file = $destination->getFileUri();
    $file_path = $this->fileSystem->realpath($uri_file);
    $handle = fopen($file_path, 'r');
    if($handle) {
      $header = fgetcsv($handle);
      $data = [];
      while (($row = fgetcsv($handle)) !== FALSE) {
        $data[] = [
          'type' => $process,
          'data' => $row,
        ];
      }
      fclose($handle);
    }
    $batch = (new BatchBuilder())
        ->addOperation([EntitiesFeedImportBatch::class, 'processData'], [$data])
        ->setTitle($this->t('Procesando datos del CSV'))
        ->setInitMessage($this->t('Iniciando procesamiento...'))
        ->setProgressMessage($this->t('Procesando @current de @total elementos.'))
        ->setErrorMessage($this->t('Error en el procesamiento.'))
        ->toArray();
    return batch_set($batch);
  }
}