<?php

/**
 * @file
 * Contains \Drupal\entities_feed\Batch\BatchProcessor.
 */

namespace Drupal\entities_feed\Batch;

use Drupal\Core\StringTranslation\StringTranslationTrait;

use function PHPSTORM_META\type;

/**
 * Clase para procesar datos en batch.
 */
class EntitiesFeedImportBatch {
  use StringTranslationTrait;

  /**
   * Procesa el array de datos en chunks.
   *
   * @param array $data
   *   El array con los datos del CSV (cada elemento es una fila).
   * @param array $context
   *   El contexto del batch (sandbox para progreso).
   */
  public static function processData(array $data, array &$context) {
    // Inicializa el sandbox si es la primera operación.
    if (!isset($context['sandbox']['progress'])) {
      $context['sandbox']['progress'] = 0;
      $context['sandbox']['max'] = count($data);
      $context['sandbox']['results']['created'] = 0;
      $context['sandbox']['results']['errors'] = 0;
      $context['results']['total'] = $context['sandbox']['max'];
    }

    $current = &$context['sandbox']['progress'];
    $max = $context['sandbox']['max'];
    $limit = 50;
    $count = 0;

    while ($current < $max && $count < $limit) {
      $row = $data[$current];

      try {
        switch ($row["type"]) {
            case 'entities':
                $response = self::create_entity($row['data']);
                break;

            case 'fields':
                $response = self::create_field($row['data']);
                break;
            
            default:
                # code...
                break;
        }
          if(!empty($response)) {
              $context['sandbox']['results']['created']++;
          } else {
            $context['sandbox']['results']['errors']++;
          }
      } catch (\Exception $e) {
        $context['sandbox']['results']['errors']++;
        \Drupal::logger('entities_feed')->error('Error procesando fila @index: @msg', [
          '@index' => $current,
          '@msg' => $e->getMessage(),
        ]);
      }

      $current++;
      $count++;
    }

    // Actualiza el progreso (valor entre 0 y 1).
    $context['finished'] = (float) $current / $max;

    // Mensaje de progreso (usa el trait para t()).
    if ($context['finished'] < 1) {
      $context['message'] = t('Procesados @current de @max elementos (creados: @created).', [
        '@current' => $current,
        '@max' => $max,
        '@created' => $context['sandbox']['results']['created'],
      ]);
    } else {
      // Mensaje final.
      $context['message'] = t('Procesamiento completado. Creados: @created, Errores: @errors.', [
        '@created' => $context['sandbox']['results']['created'],
        '@errors' => $context['sandbox']['results']['errors'],
      ]);

      // Opcional: Mensaje global al finalizar (depende de Messenger service).
      \Drupal::messenger()->addMessage($context['message']);
    }
  }

    /**
    * Create entities.
    * @param array $data
    */
    public static function create_entity($data) {
        $description = $data[2];
        $id = $data[1] ?? FALSE;
        $label = $data[0] ?? $id;
        if(!$id) {
          return NULL;
        }
        switch ($data[3]) {
            case 'content_type':            
                    $node_type = \Drupal\node\Entity\NodeType::load($id);
                    if($node_type) {
                        \Drupal::messenger()->addMessage(t('El tipo @id ya existe.', ['@id' => $id]));
                    } else {
                        $node_type = \Drupal\node\Entity\NodeType::create([
                            'type' => $id,
                            'name' => $label,
                            'description' => $description,
                            'help' => $description,
                            'new_revision' => TRUE,
                            'display_submitted' => TRUE,
                        ]);
                        $node_type->save();
                    }
                    $response = $node_type;
                break;
                
            case 'taxonomy':
                $vocabulary = \Drupal\taxonomy\Entity\Vocabulary::load($id);
                if ($vocabulary) {
                     \Drupal::messenger()->addMessage(t('El tipo @id ya existe.', ['@id' => $id]));

                } else {
                    $vocabulary = \Drupal\taxonomy\Entity\Vocabulary::create([
                        'vid' => $id,
                        'name' => $label,
                        'description' => $description,
                        'hierarchy' => 1, 
                        'weight' => 0, 
                    ]);
                    $vocabulary->save();
                }
                $response = $vocabulary;
                break;
            
            default:
                # code...
                break;
        }
        return $response;
  }

  /**
   * Create field process.
   */
  public static function create_field($data) {
    $id = $data[0] ? 'field_' . $data[0] : FALSE;
    $entity_target_id = $data[1] ?? FALSE;
    $label = $data[2] ?? $id;
    $field_type = $data[3];
    $ref_bundle = $data[4];
    $required = (bool) $data[5]; 
    $cardinality = self::getAllowed($data[6]);
    $widget = $data[7];
    $help = $data[8];
    $options = self::getOptions($data[9]);
    $response = NULL;
    if(!$entity_target_id) {
      return NULL;
    }
    $entity_target = \Drupal\node\Entity\NodeType::load($entity_target_id) ?? \Drupal\taxonomy\Entity\Vocabulary::load($entity_target_id);
    if(empty($entity_target)) {
      return NULL;
    }
    if($entity_target instanceof \Drupal\node\Entity\NodeType ) {
      $entity_target_type = 'node';

    } else {
      $entity_target_type = 'taxonomy_term';
    }
    $field_storage = self::createFieldStorage($id, $entity_target_type, $field_type, $cardinality, $options, $ref_bundle);
    if(empty($field_storage)) {
      return NULL;
    }
    if (!\Drupal\field\Entity\FieldConfig::loadByName($entity_target_type, $id, $entity_target_id)) {
      $field_config = \Drupal\field\Entity\FieldConfig::create([
        'field_name' => $id,
        'entity_type' => $entity_target_type,
        'bundle' => $entity_target_id,
        'label' => $label,
        'description' => $label,
        'required' => $required,
      ]);
      $response = $field_config->save();
    }
    return $response;
  }

  /**
   * Create fields storage.
   */
  public static function createFieldStorage($id, $entity_target_type, $field_type,          $cardinality, $options, $ref_bundle) {
    $settings = self::getFieldStorageSettings($field_type, $options, $ref_bundle);
    $field_storage = \Drupal\field\Entity\FieldStorageConfig::load($entity_target_type. ".".$id);
    if(empty($field_storage)) {
      $field_storage = \Drupal\field\Entity\FieldStorageConfig::create([
        'field_name' => $id,
        'entity_type' => $entity_target_type,
        'type' => $field_type,
        'cardinality' => $cardinality,
        'settings' => $settings,
      ]);
      $field_storage->save();
    }
    return $field_storage;
  }
  /**
   * Get options values
   * @param string $value separated with ","
   */
  public static function getOptions($value) {
    $options = explode('-', $value);
    return $options;
  }

  /**
   * Get field storage sttings.
   * @param string $field_type
   */
  public static function getFieldStorageSettings($field_type, $options = [], $ref_bundle) {
    $ref_bundle_target = \Drupal\node\Entity\NodeType::load($ref_bundle) ?? \Drupal\taxonomy\Entity\Vocabulary::load($ref_bundle);
     if($ref_bundle_target instanceof \Drupal\node\Entity\NodeType ) {
      $ref_bundle_target_type = 'node';
    } else {
      $ref_bundle_target_type = 'taxonomy_term';
    }
    switch ($field_type) {
      case 'text':
        $settings = [
          'max_length' => 255, 
          'is_ascii' => FALSE,
        ];
        break;

      case 'text_long':
        $settings = [
          'text_format' => 'plain_text',
          'allowed_formats' => ['plain_text', 'basic_html'],
        ];
        break;

      case 'integer':
        $settings = [
          'min' => '',
          'max' => '',
          'prefix' => '',
          'suffix' => '',
        ];
        break;

      case 'decimal':
        $settings = [
          'min' => '',
          'max' => '',
          'prefix' => '',
          'suffix' => '',
          'scale' => 2,
        ];
        break;

      case 'float':
        $settings = [
          'min' => '',
          'max' => '',
          'prefix' => '',
          'suffix' => '',
        ];
        break;

      case 'email':
        $settings = [
          'title' => '',
          'multiple' => FALSE,
        ];
        break;

      case 'url':
        $settings = [
          'title' => '',
          'multiple' => FALSE,
        ];
        break;

      case 'telephone':
        $settings = [
          'title' => '',
          'multiple' => FALSE,
          'number_type' => 'international',
        ];
        break;

      case 'date':
        $settings = [
          'datetime_type' => 'date',
        ];
        break;

      case 'datetime':
        $settings = [
          'datetime_type' => 'datetime',
        ];
        break;

      case 'timestamp':
        $settings = [
          'datetime_type' => 'timestamp',
        ];
        break;

      case 'entity_reference':
        $settings = [
          'target_type' => $ref_bundle_target_type,
          'handler' => 'default',
          'handler_settings' => [
            'target_bundles' => [
              $ref_bundle => $ref_bundle,
            ],
            'auto_create' => FALSE,
          ],
        ];
        break;

      case 'file': 
        $settings = [
          'file_directory' => 'uploads', 
          'file_extensions' => 'txt pdf doc jpg jpeg png svg',
          'max_filesize' => '2 MB',
          'description' => '',
        ];
        break;

      case 'image': 
        $settings = [
          'file_directory' => 'images',
          'file_extensions' => 'png gif jpg jpeg svg',
          'max_filesize' => '5 MB',
          'alt_field' => TRUE,
          'title_field' => FALSE, 
          'max_resolution' => '1200x1200',
          'min_resolution' => '100x100',
          'default_image' => [
            'uuid' => NULL,
            'alt' => '',
            'title' => '',
            'width' => NULL,
            'height' => NULL,
          ],
        ];
        break;

      case 'list_string': 
  
        $settings = [
          'allowed_values' => $options,
          'allowed_values_function' => '',
        ];
        break;

      case 'list_integer':
        $settings = [
          'allowed_values' => $options
        ];
        break;

      case 'boolean':
        $settings = [
          'on_label' => 'TRUE',
          'off_label' => 'FALSE',
          'display' => 'checkbox',
        ];
        break;

      case 'color':
        $settings = [];
        break;

      case 'range_integer':
        $settings = [
          'min' => 0,
          'max' => 100,
          'step' => 1,
        ];
        break;

      case 'range_float': 
        $settings = [
          'min' => 0.0,
          'max' => 100.0,
          'step' => 0.1,
        ];
        break;

      default:
        // Tipo no soportado: settings vacío o error.
        \Drupal::logger('entities_feed')->warning('Tipo de campo no soportado: @type', ['@type' => $field_type]);
        $settings = [];
        break;
    }
    return $settings;
  }
  /**
   * Get allowed values
   * @param string $value 0 for unlimited, "" for 1
   */
  public static function getAllowed($value) {
    switch ($value) {
      case '':
        return 1;
        break;
      
      case '0':
        return 0;

      default:
        return $value;
        break;
    }
  }
}