<?php

namespace Drupal\entity_404\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultAllowed;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Access\AccessResultReasonInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\entity_404\ToggleInterface;

/**
 * Base class for Entity 404 access checks.
 */
abstract class Entity404Base implements AccessInterface, ToggleInterface {

  /**
   * The access result reason prefix.
   */
  public const ACCESS_RESULT_REASON_PREFIX = 'Entity 404: ';

  /**
   * Indicates if the access check is enabled.
   *
   * @var bool
   */
  protected bool $enabled = TRUE;

  /**
   * Indicates teh toggle status of this check.
   *
   * @var bool
   */
  protected bool $status = TRUE;

  /**
   * Class constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface|null $config_factory
   *   The config factory.
   * @param string $settings_key
   *   The settings key.
   */
  public function __construct(?ConfigFactoryInterface $config_factory, string $settings_key = '') {
    if ($config_factory !== NULL && $settings_key !== '') {
      $this->enabled = (bool) $config_factory->get('entity_404.settings')->get($settings_key);
    }
  }

  /**
   * Check the entity route access.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(RouteMatchInterface $route_match): AccessResultInterface {
    if (!$this->isToggledOn()) {
      return AccessResult::allowed()->setCacheMaxAge(0);
    }

    // Try to get the entity.
    $parameters = $route_match->getParameters()->all();
    $entity = reset($parameters);

    if (!$entity instanceof ContentEntityInterface) {
      return AccessResult::allowed();
    }

    // Allow if not enabled.
    if (!$this->enabled) {
      return AccessResult::allowed()->addCacheTags([
        'config:entity_404.settings',
      ]);
    }

    // Run the access check.
    $result = $this->checkAccess($entity);

    // Immediatly return the result if allowed.
    if ($result instanceof AccessResultAllowed) {
      return $result;
    }

    // Set the reason.
    if ($result instanceof AccessResultReasonInterface) {
      $reason = $result->getReason() ?: static::class . ' check failed';
      $reason = self::ACCESS_RESULT_REASON_PREFIX . $reason;
      $result->setReason($reason);
    }

    // Add the settings cache tag.
    if (isset($this->settingKey)) {
      $result->addCacheTags([
        'config:entity_404.settings',
      ]);
    }

    return $result;
  }

  /**
   * Run the entity route access check.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The content entity.
   *
   * @return \Drupal\Core\Access\AccessResult
   *   The access result.
   */
  abstract protected function checkAccess(ContentEntityInterface $entity): AccessResult;

  /**
   * {@inheritdoc}
   */
  public function isToggledOn(): bool {
    return $this->status;
  }

  /**
   * {@inheritdoc}
   */
  public function toggleOn(): void {
    $this->status = TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function toggleOff(): void {
    $this->status = FALSE;
  }

}
