<?php

namespace Drupal\entity_404\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Language\Language;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Url;
use Drupal\language\Entity\ContentLanguageSettings;

/**
 * Deny access if an entity isn't translated.
 */
class HasTranslation extends Entity404Base {

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * Class constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LanguageManagerInterface $language_manager) {
    parent::__construct($config_factory, 'no_translation');

    $this->languageManager = $language_manager;
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(ContentEntityInterface $entity): AccessResult {
    // Allow access if the content language settings class isn't available.
    if (!class_exists(ContentLanguageSettings::class)) {
      return AccessResult::allowed();
    }

    // Allow access if the entity is in an untranslatable language.
    $untranslatable_languages = [
      LanguageInterface::LANGCODE_NOT_SPECIFIED,
      LanguageInterface::LANGCODE_NOT_APPLICABLE,
    ];

    if (in_array($entity->language()->getId(), $untranslatable_languages, TRUE)) {
      return AccessResult::allowed();
    }

    // Allow access if content translation isn't enabled.
    $config = ContentLanguageSettings::loadByEntityTypeBundle($entity->getEntityTypeId(), $entity->bundle());
    $content_translation = $config->getThirdPartySettings('content_translation');

    if (empty($content_translation['enabled'])) {
      return AccessResult::allowed()->addCacheableDependency($config);
    }

    // Allow access if a translation exists.
    $langcode = $this->getTranslationLangcode();

    if ($entity->hasTranslation($langcode)) {
      $result = AccessResult::allowed();
    }
    else {
      $result = AccessResult::forbidden("No $langcode translation");
    }

    // Disable caching, see self::getTranslationLangcode().
    return $result->setCacheMaxAge(0);
  }

  /**
   * Get the translation langcode.
   *
   * @return string
   *   The langcode.
   */
  protected function getTranslationLangcode(): string {
    // Check checking access on a URL object the language option isn't passed along,
    // which might cause a faulty result on e.g. the entity collection overview.
    // To solve this we use the backtrace to get the language.
    $backtrace = debug_backtrace(DEBUG_BACKTRACE_PROVIDE_OBJECT | DEBUG_BACKTRACE_IGNORE_ARGS, 15);

    foreach ($backtrace as $entry) {
      // Get the URL object.
      $object = $entry['object'] ?? NULL;

      if (!$object instanceof Url) {
        continue;
      }

      // Get the language.
      $language = $object->getOption('language');

      if ($language instanceof Language) {
        return $language->getId();
      }

      // No language was specified.
      break;
    }

    // Use the content language.
    return $this->languageManager->getCurrentLanguage(LanguageInterface::TYPE_CONTENT)->getId();
  }

}
