<?php

namespace Drupal\entity_404\EventSubscriber;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\EventSubscriber\HttpExceptionSubscriberBase;
use Drupal\Core\Routing\RouteBuildEvent;
use Drupal\Core\Routing\RoutingEvents;
use Drupal\entity_404\Access\Entity404Base;
use Symfony\Component\HttpKernel\Event\ExceptionEvent;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Event subscriber to add the entity access check and replace the 403 with a 404 response.
 */
class EntityAccessSubscriber extends HttpExceptionSubscriberBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Class constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events = parent::getSubscribedEvents();
    $events[RoutingEvents::ALTER][] = 'onRouteAlter';

    return $events;
  }

  /**
   * Act on the routes being build.
   *
   * @param \Drupal\Core\Routing\RouteBuildEvent $event
   *   The route build event.
   */
  public function onRouteAlter(RouteBuildEvent $event): void {
    foreach ($this->entityTypeManager->getDefinitions() as $definition) {
      if (!is_subclass_of($definition->getClass(), ContentEntityInterface::class)) {
        continue;
      }

      /** @var \Symfony\Component\Routing\Route|null $route */
      $route = $event->getRouteCollection()->get("entity.{$definition->id()}.canonical");

      if ($route !== NULL && $route->getDefault('_entity_form') === NULL) {
        $route->setRequirement('_entity_404', 'TRUE');
      }
    }
  }

  /**
   * Replace the 403 with a 404 response if applicable.
   *
   * @param \Symfony\Component\HttpKernel\Event\ExceptionEvent $event
   *   The exception event.
   */
  public function on403(ExceptionEvent $event): void {
    $exception = $event->getThrowable();

    if (str_starts_with($exception->getMessage(), Entity404Base::ACCESS_RESULT_REASON_PREFIX)) {
      $event->setThrowable(new NotFoundHttpException($exception->getMessage(), $exception));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function getHandledFormats(): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  protected static function getPriority(): int {
    return 100;
  }

}
