<?php

declare(strict_types=1);

namespace Drupal\entity_access_password_user_data_backend\Service;

use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\user\UserDataInterface;

/**
 * Handle access data in user data.
 */
class UserDataBackend implements UserDataBackendInterface {

  public function __construct(
    protected AccountProxyInterface $currentUser,
    protected UserDataInterface $userData,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function storeEntityAccess(FieldableEntityInterface $entity): void {
    // Do not store for anonymous user.
    if ($this->currentUser->id() == 0) {
      return;
    }

    $this->userData->set(static::MODULE_NAME, $this->currentUser->id(), $this->getEntityName($entity), TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function storeEntityBundleAccess(FieldableEntityInterface $entity): void {
    // Do not store for anonymous user.
    if ($this->currentUser->id() == 0) {
      return;
    }

    $this->userData->set(static::MODULE_NAME, $this->currentUser->id(), $this->getBundleNameFromEntity($entity), TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function storeGlobalAccess(): void {
    // Do not store for anonymous user.
    if ($this->currentUser->id() == 0) {
      return;
    }

    $this->userData->set(static::MODULE_NAME, $this->currentUser->id(), $this->getGlobalName(), TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function hasUserAccessToEntity(FieldableEntityInterface $entity): bool {
    // Do nothing for anonymous user.
    if ($this->currentUser->id() == 0) {
      return FALSE;
    }

    $has_entity_access = $this->userData->get(static::MODULE_NAME, $this->currentUser->id(), $this->getEntityName($entity));
    if ($has_entity_access) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function hasUserAccessToBundle(FieldableEntityInterface $entity): bool {
    // Do nothing for anonymous user.
    if ($this->currentUser->id() == 0) {
      return FALSE;
    }

    $has_bundle_access = $this->userData->get(static::MODULE_NAME, $this->currentUser->id(), $this->getBundleNameFromEntity($entity));
    if ($has_bundle_access) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function hasUserGlobalAccess(): bool {
    // Do nothing for anonymous user.
    if ($this->currentUser->id() == 0) {
      return FALSE;
    }

    $has_global_access = $this->userData->get(static::MODULE_NAME, $this->currentUser->id(), $this->getGlobalName());
    if ($has_global_access) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityName(FieldableEntityInterface $entity): string {
    return \sprintf(static::ENTITY_NAME_KEY, $entity->getEntityTypeId(), $entity->uuid());
  }

  /**
   * {@inheritdoc}
   */
  public function getBundleName(string $entityTypeId, string $bundleId): string {
    return \sprintf(static::BUNDLE_NAME_KEY, $entityTypeId, $bundleId);
  }

  /**
   * {@inheritdoc}
   */
  public function getBundleNameFromEntity(FieldableEntityInterface $entity): string {
    return \sprintf(static::BUNDLE_NAME_KEY, $entity->getEntityTypeId(), $entity->bundle());
  }

  /**
   * {@inheritdoc}
   */
  public function getGlobalName(): string {
    return static::GLOBAL_NAME_KEY;
  }

}
