<?php

declare(strict_types=1);

namespace Drupal\entity_access_password\Cache\Context;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\Context\CalculatedCacheContextInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\entity_access_password\Service\PasswordAccessManagerInterface;

/**
 * Defines the EntityIsProtectedCacheContext service.
 *
 * Calculated cache context ID:
 * 'entity_access_password_entity_is_protected:%entity_type_id||%entity_id||%view_mode',
 * e.g. 'entity_access_password_entity_is_protected:node||42||teaser'.
 * Or
 * 'entity_access_password_entity_is_protected:%entity_type_id||%entity_id',
 * e.g. 'entity_access_password_entity_is_protected:node||42'.
 */
class EntityIsProtectedCacheContext implements CalculatedCacheContextInterface {

  /**
   * The context ID prefix.
   */
  public const string CONTEXT_ID = 'entity_access_password_entity_is_protected';

  /**
   * View mode position when parsing the view mode context.
   */
  public const int VIEW_MODE_POSITION = 2;

  /**
   * The entries already processed.
   *
   * @var array
   */
  protected array $processedEntries = [];

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected PasswordAccessManagerInterface $passwordAccessManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getLabel() {
    return \t('Entity Access Password: Entity is protected.');
  }

  /**
   * {@inheritdoc}
   */
  public function getContext($parameter = NULL): string {
    if (isset($this->processedEntries[$parameter]['context_value'])) {
      // @phpstan-ignore-next-line
      return $this->processedEntries[$parameter]['context_value'];
    }

    // Impossible to determine the entity so do nothing.
    if ($parameter === NULL) {
      $this->processedEntries[$parameter]['context_value'] = '0';
      return $this->processedEntries[$parameter]['context_value'];
    }

    // There should be at least 2 parts.
    $parsed_entity_info = \explode('||', $parameter);
    switch (\count($parsed_entity_info)) {
      case (int) 2:
        $entity_type_id = $parsed_entity_info[0];
        $entity_id = $parsed_entity_info[1];
        $entity = $this->loadEntity($entity_type_id, $entity_id);
        if ($entity == NULL) {
          $this->processedEntries[$parameter]['context_value'] = '0';
          return $this->processedEntries[$parameter]['context_value'];
        }
        // Entity is not protected.
        if (!$this->passwordAccessManager->isEntityLabelProtected($entity)) {
          $this->processedEntries[$parameter]['context_value'] = '0';
          return $this->processedEntries[$parameter]['context_value'];
        }
        break;

      case (int) 3:
        $entity_type_id = $parsed_entity_info[0];
        $entity_id = $parsed_entity_info[1];
        $view_mode = $parsed_entity_info[static::VIEW_MODE_POSITION];
        $entity = $this->loadEntity($entity_type_id, $entity_id);
        if ($entity == NULL) {
          $this->processedEntries[$parameter]['context_value'] = '0';
          return $this->processedEntries[$parameter]['context_value'];
        }
        // Entity view mode is not protected.
        if (!$this->passwordAccessManager->isEntityViewModeProtected($view_mode, $entity)) {
          $this->processedEntries[$parameter]['context_value'] = '0';
          return $this->processedEntries[$parameter]['context_value'];
        }
        break;

      default:
        $this->processedEntries[$parameter]['context_value'] = '0';
        return $this->processedEntries[$parameter]['context_value'];
    }

    // User has access.
    if ($this->passwordAccessManager->hasUserAccessToEntity($entity)) {
      $this->processedEntries[$parameter]['context_value'] = '0';
      return $this->processedEntries[$parameter]['context_value'];
    }

    $this->processedEntries[$parameter]['context_value'] = '1';
    return $this->processedEntries[$parameter]['context_value'];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheableMetadata($parameter = NULL): CacheableMetadata {
    if (isset($this->processedEntries[$parameter]['cacheable_metadata'])) {
      // @phpstan-ignore-next-line
      return $this->processedEntries[$parameter]['cacheable_metadata'];
    }

    $this->processedEntries[$parameter]['cacheable_metadata'] = new CacheableMetadata();

    if ($parameter === NULL) {
      return $this->processedEntries[$parameter]['cacheable_metadata'];
    }

    $parsed_entity_info = \explode('||', $parameter);
    if (\count($parsed_entity_info) < (int) 2) {
      return $this->processedEntries[$parameter]['cacheable_metadata'];
    }

    $entity_type_id = $parsed_entity_info[0];
    $entity_id = $parsed_entity_info[1];
    $entity = $this->loadEntity($entity_type_id, $entity_id);
    if ($entity == NULL) {
      return $this->processedEntries[$parameter]['cacheable_metadata'];
    }

    $this->processedEntries[$parameter]['cacheable_metadata']->addCacheableDependency($entity);
    return $this->processedEntries[$parameter]['cacheable_metadata'];
  }

  /**
   * Load the fieldable entity if possible.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $entity_id
   *   The entity ID.
   *
   * @return \Drupal\Core\Entity\FieldableEntityInterface|null
   *   The fieldable entity if found. NULL otherwise.
   */
  protected function loadEntity(string $entity_type_id, string $entity_id) {
    try {
      $entity_storage = $this->entityTypeManager->getStorage($entity_type_id);
    }
    catch (\Throwable $exception) {
      return NULL;
    }

    $entity = $entity_storage->load($entity_id);
    if ($entity == NULL) {
      return NULL;
    }
    if (!$entity instanceof FieldableEntityInterface) {
      return NULL;
    }

    return $entity;
  }

}
