<?php

declare(strict_types=1);

namespace Drupal\entity_access_password\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\ConfigTarget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\ToConfig;
use Drupal\Core\Password\PasswordInterface;

/**
 * Configure Entity Access Password settings for this site.
 */
class SettingsForm extends ConfigFormBase {

  use AutowireTrait;

  /**
   * Machine name of the config.
   */
  public const string CONFIG_NAME = 'entity_access_password.settings';

  /**
   * Minimum possible length of the random password.
   */
  public const int RANDOM_PASSWORD_LENGTH_MIN = 8;

  /**
   * Maximum possible length of the random password.
   */
  public const int RANDOM_PASSWORD_LENGTH_MAX = 50;

  public function __construct(
    ConfigFactoryInterface $configFactory,
    TypedConfigManagerInterface $typedConfigManager,
    protected PasswordInterface $password,
  ) {
    parent::__construct($configFactory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_access_password_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [static::CONFIG_NAME];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['global_password'] = [
      '#type' => 'password_confirm',
      '#title' => $this->t('Global password'),
      '#description' => $this->t('If left empty will not overwrite current password (if any).'),
      '#config_target' => new ConfigTarget(
        static::CONFIG_NAME,
        'global_password',
        static fn ($value) => ['pass1' => '', 'pass2' => ''],
        fn ($value) => (\is_string($value) && !empty($value)) ? $this->password->hash($value) : ToConfig::NoOp,
      ),
      '#size' => (int) 25,
    ];
    $form['random_password_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Random password length'),
      '#description' => $this->t('The length of the randomly generated passwords.'),
      '#config_target' => static::CONFIG_NAME . ':random_password_length',
      '#required' => TRUE,
      '#min' => static::RANDOM_PASSWORD_LENGTH_MIN,
      '#max' => static::RANDOM_PASSWORD_LENGTH_MAX,
      '#step' => 1,
    ];

    return parent::buildForm($form, $form_state);
  }

}
