<?php

declare(strict_types=1);

namespace Drupal\entity_access_password\Hook;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\entity_access_password\Event\FileUsageEntityListEvent;
use Drupal\entity_access_password\Service\PasswordAccessManagerInterface;
use Drupal\file\FileUsage\FileUsageInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Access control for private files.
 */
class FileDownload {

  /**
   * Return value expected by hook_file_download when denying access.
   */
  public const int ACCESS_DENIED_RETURN = -1;

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected FileUsageInterface $fileUsage,
    protected PasswordAccessManagerInterface $passwordAccessManager,
    protected EventDispatcherInterface $eventDispatcher,
  ) {}

  /**
   * Implements hook_file_download().
   *
   * Check if the file is attached to a password protected entity.
   */
  #[Hook('file_download')]
  public function fileDownload(string $uri): array|int|null {
    /** @var \Drupal\file\FileInterface[] $files */
    $files = $this->entityTypeManager->getStorage('file')->loadByProperties([
      'uri' => $uri,
    ]);

    // There should be only one file per URI.
    foreach ($files as $file) {
      $usages = $this->fileUsage->listUsage($file);
      $user_has_access = NULL;

      foreach ($usages as $entity_list) {
        /** @var array $entity_list */
        // It is not possible without custom development or modules like
        // Entity Browser to re-use a file in multiple entities so there should
        // be most of the time only one entity.
        foreach ($entity_list as $entity_type_id => $entity_ids) {
          /** @var array $entity_ids */
          $entity_storage = $this->entityTypeManager->getStorage($entity_type_id);
          $entities = $entity_storage->loadMultiple(\array_keys($entity_ids));

          // Allows to alter the list of entities.
          $file_usage_entity_list_event = new FileUsageEntityListEvent($file, $entities);
          $this->eventDispatcher->dispatch($file_usage_entity_list_event);

          foreach ($file_usage_entity_list_event->getEntities() as $entity) {
            $user_has_access = $this->passwordAccessManager->hasUserAccessToEntity($entity);
            // If the user has access to at least one entity using the file do
            // nothing.
            if ($user_has_access) {
              return NULL;
            }
          }
        }
      }

      // It means that only password protected entities had been encountered
      // and that the user has access to none.
      if ($user_has_access === FALSE) {
        return static::ACCESS_DENIED_RETURN;
      }
    }

    return NULL;
  }

}
