<?php

declare(strict_types=1);

namespace Drupal\entity_access_password\Hook;

use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\entity_access_password\Cache\Context\EntityIsProtectedCacheContext;
use Drupal\entity_access_password\Service\PasswordAccessManagerInterface;

/**
 * Declare a new token for entities.
 */
class Token {

  use StringTranslationTrait;

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected PasswordAccessManagerInterface $passwordAccessManager,
  ) {}

  /**
   * Implements hook_token_info().
   */
  #[Hook('token_info')]
  public function tokenInfo(): array {
    $info = [];

    $entities = $this->entityTypeManager->getDefinitions();
    foreach ($entities as $entity_info) {
      // Do not generate tokens if the entity doesn't define a token type or is
      // not a content entity.
      if (!$entity_info->get('token_type') || (!$entity_info instanceof ContentEntityTypeInterface)) {
        continue;
      }

      /** @var string $token_type */
      $token_type = $entity_info->get('token_type');

      $entity_type_label = $entity_info->getLabel();
      if ($entity_type_label instanceof TranslatableMarkup) {
        $entity_type_label = $entity_type_label->__toString();
      }
      $entity_type_label = \mb_strtolower($entity_type_label);

      // Add [entity:protected-label] tokens.
      $info['tokens'][$token_type]['protected-label'] = [
        'name' => $this->t('Protected label'),
        'description' => $this->t('The label of the @entity if the user has access.', [
          '@entity' => $entity_type_label,
        ]),
        'module' => 'entity_access_password',
      ];
    }

    return $info;
  }

  /**
   * Implements hook_tokens().
   */
  #[Hook('tokens')]
  public function tokens(string $type, array $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata): array {
    $replacements = [];

    if ($type == 'entity') {
      foreach ($tokens as $name => $original) {
        /** @var string $original */
        switch ($name) {
          case 'protected-label':
            /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
            $entity = $data['entity'];
            $replacements[$original] = $entity->label();

            if ($this->passwordAccessManager->isEntityLabelProtected($entity)) {
              $bubbleable_metadata->addCacheContexts([EntityIsProtectedCacheContext::CONTEXT_ID . ':' . $entity->getEntityTypeId() . '||' . $entity->id()]);

              if (!$this->passwordAccessManager->hasUserAccessToEntity($entity)) {
                $replacements[$original] = $this->t('Protected entity');
              }
            }
            break;
        }
      }
    }

    return $replacements;
  }

}
