<?php

declare(strict_types=1);

namespace Drupal\entity_access_password\Hook;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\entity_access_password\Cache\Context\EntityIsProtectedCacheContext;
use Drupal\entity_access_password\Service\PasswordAccessManagerInterface;

/**
 * Replace the view mode by the protected view mode if needed.
 */
class ViewModeAlter {

  public function __construct(
    protected PasswordAccessManagerInterface $passwordAccessManager,
  ) {}

  /**
   * Implements hook_entity_view_mode_alter().
   */
  #[Hook('entity_view_mode_alter')]
  public function alter(string &$view_mode, EntityInterface $entity): void {
    // Quick return to avoid instantiation.
    if ($view_mode == PasswordAccessManagerInterface::PROTECTED_VIEW_MODE) {
      return;
    }

    if ($this->passwordAccessManager->isEntityViewModeProtected($view_mode, $entity)) {
      $entity->addCacheContexts([EntityIsProtectedCacheContext::CONTEXT_ID . ':' . $entity->getEntityTypeId() . '||' . $entity->id() . '||' . $view_mode]);

      if (!$this->passwordAccessManager->hasUserAccessToEntity($entity)) {
        $view_mode = PasswordAccessManagerInterface::PROTECTED_VIEW_MODE;
      }
    }
  }

}
