<?php

declare(strict_types=1);

namespace Drupal\entity_access_password\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Password\PasswordInterface;
use Drupal\entity_access_password\Form\SettingsForm;
use Drupal\entity_access_password\Plugin\Field\FieldType\EntityAccessPasswordItem;

/**
 * The default password validator.
 */
class PasswordValidator implements PasswordValidatorInterface {

  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected PasswordInterface $password,
    protected AccessStorageInterface $accessStorage,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function validatePassword(string $password, EntityAccessPasswordItem $fieldItem): bool {
    /** @var array{password_entity: bool, password_bundle: bool, password_global: bool, password: string, } $field_instance_settings */
    $field_instance_settings = $fieldItem->getFieldDefinition()->getSettings();

    // Entity password.
    if ($field_instance_settings['password_entity']) {
      /** @var array{password: string} $values */
      $values = $fieldItem->getValue();
      if (!empty($values['password']) && $this->password->check($password, $values['password'])) {
        $entity = $fieldItem->getEntity();
        $this->accessStorage->storeEntityAccess($entity);
        return TRUE;
      }
    }

    // Bundle password.
    if ($field_instance_settings['password_bundle']) {
      if (!empty($field_instance_settings['password']) && $this->password->check($password, $field_instance_settings['password'])) {
        $entity = $fieldItem->getEntity();
        $this->accessStorage->storeEntityBundleAccess($entity);
        return TRUE;
      }
    }

    // Global password.
    if ($field_instance_settings['password_global']) {
      $config = $this->configFactory->get(SettingsForm::CONFIG_NAME);
      /** @var string $global_password */
      $global_password = $config->get('global_password');
      if (!empty($global_password) && $this->password->check($password, $global_password)) {
        $this->accessStorage->storeGlobalAccess();
        return TRUE;
      }
    }

    return FALSE;
  }

}
