<?php

declare(strict_types=1);

namespace Drupal\entity_access_password_user_data_backend\Hook;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\entity_access_password_user_data_backend\Routing\EntityFormRoutes;

/**
 * Manipulates entity type information.
 */
class EntityTypeInfo {

  use StringTranslationTrait;

  /**
   * The access permission.
   */
  public const string ACCESS_PERMISSION = 'entity_access_password_user_data_backend_access_entity_form';

  /**
   * The entity operation. Also used for the dynamic route and task link.
   */
  public const string ENTITY_OPERATION = 'entity_access_password_user_data_edit';

  /**
   * The entity operation weight.
   */
  public const int ENTITY_OPERATION_WEIGHT = 50;

  public function __construct(
    protected AccountProxyInterface $currentUser,
  ) {}

  /**
   * Implements hook_entity_operation().
   *
   * Adds operations on entity that supports it.
   */
  #[Hook('entity_operation')]
  public function entityOperation(EntityInterface $entity): array {
    $operations = [];
    if (
      $entity instanceof FieldableEntityInterface
      && $this->currentUser->hasPermission(static::ACCESS_PERMISSION)
      && $entity->access('update')
    ) {
      $fields = $entity->getFields();
      foreach ($fields as $field) {
        $field_definition = $field->getFieldDefinition();
        if ($field_definition->getType() == 'entity_access_password_password') {
          $entity_type_id = $entity->getEntityTypeId();
          $bundle_id = $entity->bundle();
          $route_name = \sprintf(EntityFormRoutes::ROUTE_NAME, $entity_type_id, $bundle_id);
          $operations[static::ENTITY_OPERATION] = [
            'title' => $this->t('Password access user data'),
            'url' => Url::fromRoute($route_name, [
              $entity_type_id => $entity->id(),
            ]),
            'weight' => static::ENTITY_OPERATION_WEIGHT,
          ];
          break;
        }
      }
    }
    return $operations;
  }

}
