# Entity Attributes

A Drupal module that provides a flexible system for managing HTML attributes (`id`, `class`, `data-*`, etc.) across various entity types through an admin UI.

## Overview

This module allows site administrators and content editors to add custom HTML attributes to entities that will be automatically applied in Twig templates during rendering. It supports both content entities (nodes, taxonomy terms, menu links) and configuration entities (blocks, menus) through a unified plugin architecture.

## Features

### Core Functionality

- **Plugin Architecture**: Extensible plugin system supporting multiple entity types
- **Dynamic Field Management**: Automatically adds/removes attribute fields when plugins are enabled/disabled
- **YAML-based Configuration**: Human-readable attribute storage with built-in validation
- **Permission System**: Granular permissions per entity type and bundle
- **Smart Form Integration**: Seamlessly integrates with existing entity forms
- **CodeMirror Editor Integration**: Enhanced UX for editing attributes

### Supported Entity Types

- Nodes
- Taxonomy Terms
- Blocks
- Menus
- Content Menu Links
- Static Menu Links
- Paragraphs
- ECK (Entity Construction Kit) entities

Easy to add support for additional entity types.

### Supported Attributes

The module doesn't apply any restrictions on the attribute name and values. Any valid attribute name and value entered in the attributes field will be converted to an attributes object and passed to the entity template.

### Attribute Sets

Each entity type may support multiple attribute sets, used in their templates:

- `attributes`: Main container attributes (`id`, `class`, `data-*`)
- `title_attributes`: Title-specific attributes
- `content_attributes`: Content wrapper attributes
- Plugin-specific attribute sets

## Installation

1. Install as you would normally install a contributed Drupal module. See [Installing Modules](https://www.drupal.org/docs/extending-drupal/installing-modules) for more details.

2. Enable the module.

3. Configure enabled entity types at **Configuration → Search and metadata → Entity Attributes** (`/admin/config/search/entity-attributes`).

4. Configure permissions at **People → Permissions** (`/admin/people/permissions`), filter by "Entity Attributes".

## Configuration

### Enabling Entity Types

1. Navigate to **Configuration → Search and metadata → Entity Attributes** (`/admin/config/search/entity-attributes`)
2. Select which entity types and bundles should have attribute fields
3. Save the configuration
4. Fields will be automatically added to enabled entities

## Usage

When enabled, an "Attributes" field appears in entity edit forms, an example of field value:

```yaml
attributes:
  id: 'my-custom-id'
  class:
    - 'featured-content'
    - 'highlight'
  data-toggle: 'modal'
  data-target: '#myModal'

title_attributes:
  class:
    - 'custom-title'

content_attributes:
  role: 'region'
  aria-label: 'Main content area'
```

## Template Integration

Attributes are automatically available in Twig templates.

An example of using attributes in a node template:

```twig
{# web/themes/custom/mytheme/templates/content/node--full.html.twig #}

<article{{ attributes }}>
  <h1{{ title_attributes }}>{{ label }}</h1>
  <div{{ content_attributes }}>
    {{ content }}
  </div>
</article>
```

Some themes may not apply all the attributes set in some templates. It is quite easy to fix by adding them manually in your custom theme template overrides.

An example of adding link_attributes for menu items:

```twig
{# web/themes/custom/mytheme/templates/naviagation/menu.html.twig #}

{% import _self as menus %}
{{ menus.menu_links(items, attributes, 0) }}

{% macro menu_links(items, attributes, menu_level) %}
  {% import _self as menus %}
  {% if items %}
    {% if menu_level == 0 %}
      <ul{{ attributes }}>
    {% else %}
      <ul>
    {% endif %}
    {% for item in items %}
      <li{{ item.attributes }}>
        {# Adding link_attributes to the link #}
        {{ link(item.title, item.url, item.link_attributes) }}
        {% if item.below %}
          {{ menus.menu_links(item.below, attributes, menu_level + 1) }}
        {% endif %}
      </li>
    {% endfor %}
    </ul>
  {% endif %}
{% endmacro %}
```

## Architecture

In general, the attributes functionality is already supported in Drupal core. Despite actively using attributes in entity templates, currently, there is no UI for managing attributes for a particular entity. This module extends Cores' functionality by providing an attributes field, storing their values, and passing them to templates.

The support of attributes for each entity type or bundle includes three parts:

* An Entity Attributes plugin. Handles adding, removing, and storage of the attributes field for the entity type or bundle.
  The module provides two base classes:
  - `ContentEntityAttributesBase`: For content entities (nodes, terms, content menu links)
  - `ConfigEntityAttributesBase`: For configuration entities (blocks, menus, static menu links)
* The attributes field display. Could be handled by widget element or added by the corresponding form alter hook.
* The template preprocess. Implemented via preprocess hook.

### Adding attributes to other entities

Adding the attributes support for the entity type is relatively easy:

* Create a new EntityAttributes plugin in your custom module.
* You can extend existing base classes or create your own, implementing `EntityAttributesInterface`
* For content entities: add entity form customizations if needed.
* For config or custom entities: provide a handler that adds the attributes field to the entity form.
* Implement entity preprocess handler.

## Development

Planned features:

* ~~Add CodeMirror editor support for the attributes field;~~ Done
* ~~Entity Attributes plugin for Paragraphs;~~ Done
* ~~Entity Attributes ECK (Entity Construction Kit) entities;~~ Done
* Test coverage.

Contributions are welcome.
