<?php

namespace Drupal\entity_attributes;

/**
 * Base class for Config Entity Attributes plugins.
 *
 * This class handles config entities that use third_party_settings
 * for storing attributes (like blocks).
 */
abstract class ConfigEntityAttributesBase extends EntityAttributesBase {

  /**
   * {@inheritdoc}
   */
  public function hasAttributesField(string $entity_type, string $bundle): bool {
    // Config entities don't use fields, check the config instead.
    $config = $this->configFactory->get('entity_attributes.settings');
    $enabled_bundles = $config->get('enabled_bundles.' . $entity_type) ?: [];

    return \in_array($bundle, $enabled_bundles, TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function addAttributesField(string $entity_type, string $bundle): bool {
    // For config entities, we don't need to add fields.
    // The attributes are handled via third_party_settings.
    // Just verify the bundle is valid and log the action.
    $entity_type_label = $this->getEntityTypeLabel($entity_type);
    $bundles = $this->getBundles($entity_type);
    if (!isset($bundles[$bundle])) {
      $this->logger->error('Invalid bundle @bundle for @entity_type entity', [
        '@bundle' => $bundles[$bundle],
        '@entity_type' => $entity_type_label,
      ]);
      return FALSE;
    }

    $this->logger->info('@entity_type attributes support enabled for @bundle', [
      '@entity_type' => $entity_type_label,
      '@bundle' => $bundles[$bundle],
    ]);

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function removeAttributesField(string $entity_type, string $bundle): bool {
    // For config entities, removal would involve clearing third_party_settings
    // from existing instances.
    $entity_type_label = $this->getEntityTypeLabel($entity_type);
    $bundles = $this->getBundles($entity_type);
    try {
      // Load all entities of this type and clear their third_party_settings.
      $entity_storage = $this->entityTypeManager->getStorage($entity_type);
      $entities = $entity_storage->loadMultiple();

      $cleaned_count = 0;
      foreach ($entities as $entity) {
        /** @var \Drupal\Core\Config\Entity\ConfigEntityInterface $entity */
        if ($entity->getThirdPartySetting('entity_attributes', 'attributes_data')) {
          $entity->unsetThirdPartySetting('entity_attributes', 'attributes_data');
          $entity->save();
          $cleaned_count++;
        }
      }

      if ($cleaned_count > 0) {
        $this->logger->info('Cleaned third_party_settings from @count @entity_type entities', [
          '@count' => $cleaned_count,
          '@entity_type' => $entity_type_label,
        ]);
      }

      $this->logger->info('@entity_type attributes support disabled for @bundle', [
        '@entity_type' => $entity_type_label,
        '@bundle' => $bundles[$bundle],
      ]);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to clean up @entity_type third_party_settings: @error', [
        '@entity_type' => $entity_type_label,
        '@error' => $e->getMessage(),
      ]);

      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function updateFieldStorageLockStatus(string $entity_type): void {
    // Configuration entities don't use field storage,
    // so this method is a no-op.
  }

}
