<?php

namespace Drupal\entity_attributes;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;

/**
 * Base class for Content Entity Attributes plugins.
 *
 * This class handles content entities that use fields for storing attributes
 * (like nodes and taxonomy terms).
 */
abstract class ContentEntityAttributesBase extends EntityAttributesBase {

  /**
   * {@inheritdoc}
   */
  public function hasAttributesField(string $entity_type, string $bundle): bool {
    $field_config = FieldConfig::loadByName($entity_type, $bundle, $this->getFieldName());
    return !empty($field_config);
  }

  /**
   * {@inheritdoc}
   */
  public function addAttributesField(string $entity_type, string $bundle): bool {
    if ($this->hasAttributesField($entity_type, $bundle)) {
      return TRUE;
    }

    $field_name = $this->getFieldName();
    $entity_type_label = $this->getEntityTypeLabel($entity_type);
    $bundles = $this->getBundles($entity_type);
    $bundle_label = $bundles[$bundle];
    try {
      // Create field storage if it doesn't exist
      // (shared across all bundles of this entity type).
      $field_storage = FieldStorageConfig::loadByName($entity_type, $field_name);
      if (!$field_storage) {
        $field_storage = FieldStorageConfig::create([
          'field_name' => $field_name,
          'entity_type' => $entity_type,
          'type' => 'string_long',
          'cardinality' => 1,
          'locked' => TRUE,
          'persist_with_no_fields' => TRUE,
          'settings' => [
            'case_sensitive' => FALSE,
          ],
        ]);
        $field_storage->save();

        $this->logger->info(
          'Created @field_name field storage for entity type @entity_type', [
            '@field_name' => $this->t('Attributes'),
            '@entity_type' => $entity_type_label,
          ]);
      }

      // Create a field instance for this specific bundle.
      $field_config = FieldConfig::create([
        'field_storage' => $field_storage,
        'bundle' => $bundle,
        'label' => $this->t('Attributes'),
        'required' => FALSE,
        'settings' => [
          'allowed_formats' => [],
        ],
      ]);

      // Set field settings based on an entity type.
      $this->configureField($field_config);
      $field_config->save();
      $this->configureFormDisplay($entity_type, $bundle);
      $this->configureViewDisplay($entity_type, $bundle);
      $this->updateFieldStorageLockStatus($entity_type);

      $this->logger->info('Added @field_name field to @entity_type - @bundle', [
        '@field_name' => $this->t('Attributes'),
        '@entity_type' => $entity_type_label,
        '@bundle' => $bundle_label,
      ]);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to add @field_name field to @entity_type - @bundle: @error', [
        '@field_name' => $this->t('Attributes'),
        '@entity_type' => $entity_type_label,
        '@bundle' => $bundle_label,
        '@error' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function removeAttributesField(string $entity_type, string $bundle): bool {
    $field_name = $this->getFieldName();

    $field_config = FieldConfig::loadByName($entity_type, $bundle, $field_name);
    if (!$field_config) {
      // Field doesn't exist for this bundle, consider it successfully removed.
      return TRUE;
    }

    $entity_type_label = $this->getEntityTypeLabel($entity_type);
    $bundles = $this->getBundles($entity_type);
    $bundle_label = $bundles[$bundle];
    try {
      // Delete the field instance for this specific bundle.
      $field_config->delete();

      $this->logger->info(
        'Removed @field_name field from @entity_type - @bundle', [
          '@field_name' => $this->t('Attributes'),
          '@entity_type' => $entity_type_label,
          '@bundle' => $bundle_label,
        ]);

      // Clear entity cache to ensure the field is properly removed.
      $this->entityTypeManager->clearCachedDefinitions();
      // Check if field storage should be deleted (if no other bundles use it).
      $this->cleanupFieldStorage($entity_type);
      // Update the field storage lock status based on configuration.
      $this->updateFieldStorageLockStatus($entity_type);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to remove @field_name field from @entity_type - @bundle: @error', [
        '@field_name' => $this->t('Attributes'),
        '@entity_type' => $entity_type_label,
        '@bundle' => $bundle_label,
        '@error' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function updateFieldStorageLockStatus(string $entity_type): void {
    $field_name = $this->getFieldName();

    $field_storage = FieldStorageConfig::loadByName($entity_type, $field_name);
    if (!$field_storage) {
      return;
    }

    $config = $this->configFactory->get('entity_attributes.settings');
    $enabled_bundles = $config->get('enabled_bundles') ?: [];
    $entity_type_label = $this->getEntityTypeLabel($entity_type);

    // Lock the field if it's managed by our configuration system.
    $is_managed = !empty($enabled_bundles[$entity_type]);

    if ($field_storage->isLocked() !== $is_managed) {
      $field_storage->set('locked', $is_managed);
      $field_storage->save();

      $this->logger->info('Updated @field_name field storage lock status for @entity_type to @status', [
        '@field_name' => $this->t('Attributes'),
        '@entity_type' => $entity_type_label,
        '@status' => $is_managed ? 'locked' : 'unlocked',
      ]);
    }
  }

  /**
   * Clean up field storage if no field instances remain.
   *
   * This implements standard Drupal behavior: field storage is shared across
   * all bundles of an entity type and should only be deleted when no bundles
   * are using it anymore.
   *
   * @param string $entity_type
   *   The entity type ID.
   */
  protected function cleanupFieldStorage(string $entity_type): void {
    $field_name = $this->getFieldName();

    // Check if there are any remaining field instances across all bundles.
    $field_configs = $this->entityTypeManager
      ->getStorage('field_config')
      ->loadByProperties([
        'entity_type' => $entity_type,
        'field_name' => $field_name,
      ]);

    $entity_type_label = $this->getEntityTypeLabel($entity_type);

    // If no field instances remain across any bundle,
    // delete the field storage.
    if (empty($field_configs)) {
      $field_storage = FieldStorageConfig::loadByName($entity_type, $field_name);
      if ($field_storage) {
        try {
          $field_storage->delete();

          $this->logger->info('Deleted @field_name field storage for entity type @entity_type (no bundles using it)', [
            '@field_name' => $this->t('Attributes'),
            '@entity_type' => $entity_type_label,
          ]);
        }
        catch (\Exception $e) {
          $this->logger->warning('Could not delete @field_name field storage for @entity_type: @error', [
            '@field_name' => $this->t('Attributes'),
            '@entity_type' => $entity_type_label,
            '@error' => $e->getMessage(),
          ]);
        }
      }
    }
    else {
      $this->logger->info('@field_name field storage for @entity_type preserved (still used by @count bundles)', [
        '@field_name' => $this->t('Attributes'),
        '@entity_type' => $entity_type_label,
        '@count' => \count($field_configs),
      ]);
    }
  }

  /**
   * Configure the field instance.
   *
   * @param \Drupal\field\Entity\FieldConfig $field_config
   *   The field configuration entity.
   */
  protected function configureField(FieldConfig $field_config): void {
    // The default implementation does nothing.
  }

  /**
   * Configure the form display for the field.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param string $bundle
   *   The bundle machine name.
   */
  protected function configureFormDisplay(string $entity_type, string $bundle): void {
    $form_display_storage = $this->entityTypeManager->getStorage('entity_form_display');
    $form_display = $form_display_storage->load($entity_type . '.' . $bundle . '.default');

    // Create a form display if it doesn't exist.
    if (!$form_display) {
      $form_display = $form_display_storage->create([
        'targetEntityType' => $entity_type,
        'bundle' => $bundle,
        'mode' => 'default',
        'status' => TRUE,
      ]);
    }

    $form_display->setComponent($this->getFieldName(), [
      'type' => 'entity_attributes_yaml',
      'weight' => 100,
      'region' => 'content',
    ]);
    $form_display->save();
  }

  /**
   * Configure the view display for the field.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param string $bundle
   *   The bundle machine name.
   */
  protected function configureViewDisplay(string $entity_type, string $bundle): void {
    $view_display_storage = $this->entityTypeManager->getStorage('entity_view_display');
    $view_display = $view_display_storage->load($entity_type . '.' . $bundle . '.default');

    // Create a view display if it doesn't exist.
    if (!$view_display) {
      $view_display = $view_display_storage->create([
        'targetEntityType' => $entity_type,
        'bundle' => $bundle,
        'mode' => 'default',
        'status' => TRUE,
      ]);
    }

    // Hide the field from view displays by default.
    $view_display->removeComponent($this->getFieldName());
    $view_display->save();
  }

}
