<?php

namespace Drupal\entity_attributes;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for Entity Attributes plugins.
 */
abstract class EntityAttributesBase extends PluginBase implements EntityAttributesInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity type bundle info service.
   */
  protected EntityTypeBundleInfoInterface $entityTypeBundleInfo;

  /**
   * The logger service.
   */
  protected LoggerInterface $logger;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->entityTypeBundleInfo = $container->get('entity_type.bundle.info');
    $instance->logger = $container->get('logger.factory')->get('entity_attributes');
    $instance->configFactory = $container->get('config.factory');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityTypes(): array {
    return $this->pluginDefinition['entity_types'] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityTypeLabel(string $entity_type): string {
    $entity_type_definition = $this->entityTypeManager->getDefinition($entity_type);

    return $entity_type_definition->getLabel();
  }

  /**
   * {@inheritdoc}
   */
  public function getBundles(string $entity_type): array {
    $bundles = $this->entityTypeBundleInfo->getBundleInfo($entity_type);
    $bundle_options = [];
    foreach ($bundles as $bundle_id => $bundle_info) {
      $bundle_options[$bundle_id] = $bundle_info['label'];
    }

    return $bundle_options;
  }

  /**
   * {@inheritdoc}
   */
  public function getFieldName(): string {
    return 'entity_attributes';
  }

  /**
   * {@inheritdoc}
   */
  abstract public function hasAttributesField(string $entity_type, string $bundle): bool;

  /**
   * {@inheritdoc}
   */
  abstract public function addAttributesField(string $entity_type, string $bundle): bool;

  /**
   * {@inheritdoc}
   */
  abstract public function removeAttributesField(string $entity_type, string $bundle): bool;

  /**
   * {@inheritdoc}
   */
  abstract public function updateFieldStorageLockStatus(string $entity_type): void;

  /**
   * {@inheritdoc}
   */
  public function getSupportedAttributeSets(): array {
    return ['attributes'];
  }

}
