<?php

declare(strict_types=1);

namespace Drupal\entity_attributes;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\Core\TypedData\TypedDataManagerInterface;

/**
 * Service for handling entity attributes field operations.
 */
class EntityAttributesField {

  use StringTranslationTrait;

  /**
   * The entity attributes plugin manager.
   */
  protected EntityAttributesManager $pluginManager;

  /**
   * The typed data manager.
   */
  protected TypedDataManagerInterface $typedDataManager;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The route match service.
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * The entity attributes permissions service.
   */
  protected EntityAttributesPermissions $permissions;

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * Constructs an EntityAttributesField object.
   *
   * @param \Drupal\entity_attributes\EntityAttributesManager $plugin_manager
   *   The entity attributes plugin manager.
   * @param \Drupal\Core\TypedData\TypedDataManagerInterface $typed_data_manager
   *   The typed data manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match service.
   * @param \Drupal\entity_attributes\EntityAttributesPermissions $permissions
   *   The entity attributes permissions service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(
    EntityAttributesManager $plugin_manager,
    TypedDataManagerInterface $typed_data_manager,
    ConfigFactoryInterface $config_factory,
    RouteMatchInterface $route_match,
    EntityAttributesPermissions $permissions,
    ModuleHandlerInterface $module_handler,
  ) {
    $this->pluginManager = $plugin_manager;
    $this->typedDataManager = $typed_data_manager;
    $this->configFactory = $config_factory;
    $this->routeMatch = $route_match;
    $this->permissions = $permissions;
    $this->moduleHandler = $module_handler;
  }

  /**
   * Generates the default YAML value for the given attribute sets.
   *
   * @param array $attribute_sets
   *   The supported attribute sets.
   * @param mixed $current_value
   *   The current value (optional).
   *
   * @return string
   *   The YAML-encoded default value.
   */
  protected function generateDefaultYamlValue(array $attribute_sets, mixed $current_value = NULL): string {
    // If we have a current value, use it,
    // otherwise create a default structure.
    if (!empty($current_value)) {
      $yaml = Yaml::encode($current_value);
    }
    else {
      $default_value = [];
      foreach ($attribute_sets as $attribute_set) {
        $default_value[$attribute_set] = [];
      }
      $yaml = Yaml::encode($default_value);
    }

    // Clean up empty arrays for better readability.
    return \str_replace('{  }', '[]', $yaml);
  }

  /**
   * Generates help text description for supported attribute sets.
   *
   * @param array $attribute_sets
   *   The supported attribute sets.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The translated description.
   */
  protected function generateAttributeSetsDescription(array $attribute_sets): TranslatableMarkup {
    $formatted_values = [];
    foreach ($attribute_sets as $attribute_set) {
      $formatted_values[] = "<code>'" . $attribute_set . "'</code>";
    }

    return $this->t('Supported attribute sets: @attribute_sets', [
      '@attribute_sets' => Markup::create(\implode(', ', $formatted_values)),
    ]);
  }

  /**
   * Creates a textarea element.
   *
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup|string $title
   *   The field title.
   * @param string $default_value
   *   The default value.
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup $description
   *   The field description.
   * @param array $parents
   *   The parents array for the field.
   * @param array $additional_properties
   *   Additional form element properties.
   *
   * @return array
   *   The form element array.
   */
  protected function createYamlTextarea(
    TranslatableMarkup|string $title,
    string $default_value,
    TranslatableMarkup $description,
    array $parents = [],
    array $additional_properties = [],
  ): array {
    $config = $this->configFactory->get('entity_attributes.settings');
    $use_codemirror = $config->get('use_codemirror') && $this->moduleHandler->moduleExists('codemirror_editor');

    $element = [
      '#title' => $title,
      '#description' => $description,
      '#default_value' => $default_value,
      '#element_validate' => [
        [$this, 'validateConfigEntityYaml'],
      ],
    ] + $additional_properties;

    if (!empty($parents)) {
      $element['#parents'] = $parents;
    }

    if ($use_codemirror) {
      $element['#type'] = 'codemirror';
      $element['#rows'] = 4;
      $element['#codemirror'] = [
        'mode' => 'text/x-yaml',
        'toolbar' => FALSE,
      ];
    }
    else {
      $element['#type'] = 'textarea';
      $element['#rows'] = 8;
    }

    return $element;
  }

  /**
   * Gets the current YAML value for a config entity attributes field.
   *
   * @param \Drupal\Core\Config\Entity\ConfigEntityInterface $entity
   *   The config entity.
   * @param array $attribute_sets
   *   The supported attribute sets.
   *
   * @return string
   *   The YAML-encoded current value or default structure.
   */
  public function getConfigEntityYamlValue(ConfigEntityInterface $entity, array $attribute_sets): string {
    $current_value = $entity->getThirdPartySetting('entity_attributes', 'attributes_data', []);

    return $this->generateDefaultYamlValue($attribute_sets, $current_value);
  }

  /**
   * Validates YAML syntax for the config entity attributes field.
   *
   * @param array $element
   *   The form element to validate.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param array $complete_form
   *   The complete form structure.
   */
  public function validateConfigEntityYaml(array &$element, FormStateInterface $form_state, array &$complete_form): void {
    if (empty($element['#value'])) {
      return;
    }

    // Using validation constraint for consistency.
    $data_definition = DataDefinition::create('string');
    $data_definition->addConstraint('ValidYaml', []);
    $typed_data = $this->typedDataManager->create($data_definition, $element['#value']);
    $violations = $typed_data->validate();

    if (\count($violations) > 0) {
      foreach ($violations as $violation) {
        $form_state->setError($element, $violation->getMessage());
      }
    }
  }

  /**
   * Submit handler for the config entity attributes field.
   *
   * Converts YAML string to array for storage in the config entity.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function submitConfigEntityYaml(array &$form, FormStateInterface $form_state): void {
    $yaml_value = $form_state->getValue(['third_party_settings', 'entity_attributes', 'attributes_data']);
    if (empty($yaml_value)) {
      return;
    }

    // The value should be already valid YAML at this point.
    $decoded = Yaml::decode($yaml_value);
    $form_state->setValue(['third_party_settings', 'entity_attributes', 'attributes_data'], $decoded);
  }

  /**
   * Submit handler for static menu link attributes.
   *
   * Converts YAML string to array for storage in the config
   * provided by the plugin.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function submitStaticMenuLinkYaml(array &$form, FormStateInterface $form_state): void {
    // Get the menu link plugin from the route.
    $menu_link = $this->routeMatch->getParameter('menu_link_plugin');
    if (empty($menu_link)) {
      return;
    }

    // The ID from menu link plugin.
    $plugin_id = $menu_link->getPluginId();
    // Entity Attributes static menu link plugin instance.
    $plugin = $this->pluginManager->createInstance('static_menu_link');
    if (!$plugin) {
      return;
    }

    $yaml_value = $form_state->getValue(['third_party_settings', 'entity_attributes', 'attributes_data']);
    $attributes = Yaml::decode($yaml_value);
    $plugin->setAttributes($plugin_id, $attributes);
  }

  /**
   * Enhances a widget element for the entity attributes field.
   *
   * @param array $element
   *   The widget element to enhance.
   * @param array $attribute_sets
   *   The supported attribute sets.
   *
   * @return array
   *   The enhanced element.
   */
  public function enhanceWidgetElement(array $element, array $attribute_sets): array {
    if (empty($element['#default_value'])) {
      $element['#default_value'] = $this->generateDefaultYamlValue($attribute_sets);
    }

    $element['#description'] = $this->generateAttributeSetsDescription($attribute_sets);

    return $element;
  }

  /**
   * Applies enhancements to the attributes field in a widget element.
   *
   * The field itself is added by the plug-in. Used in the following plugins:
   * Node, Taxonomy term, Menu link Content.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $context
   *   The context array.
   */
  public function alterFieldWidgetSingleElementForm(array &$element, FormStateInterface $form_state, array $context): void {
    // Checking that this is an entity_attributes field.
    $field_name = $context['items']->getFieldDefinition()->getName();
    if ($field_name !== 'entity_attributes') {
      return;
    }

    $entity = $context['items']->getEntity();
    if (!$entity->getEntityType()->entityClassImplements(FieldableEntityInterface::class)) {
      return;
    }

    if (!$this->pluginManager->isPluginEnabled($entity->getEntityTypeId(), $entity->bundle())) {
      return;
    }

    if (empty($element['value'])) {
      return;
    }

    // Hiding the field for users without edit permission.
    if (!$this->permissions->hasPermission($entity->getEntityTypeId(), $entity->bundle())) {
      $element['#access'] = FALSE;
    }

    // Apply enhancements to the value element (works for any widget).
    $plugin = $this->pluginManager->getPluginForEntityType($entity->getEntityTypeId());
    $attribute_sets = $plugin->getSupportedAttributeSets();
    $element['value'] = $this->enhanceWidgetElement($element['value'], $attribute_sets);
  }

  /**
   * Applies additional adjustments to the attributes field.
   *
   * By default, Menu Link Content is not a fieldable entity, and it has
   * a different form structure.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $form_id
   *   The form ID.
   */
  public function alterMenuLinkContentForm(array &$form, FormStateInterface $form_state, string $form_id): void {
    if (!$this->pluginManager->isPluginEnabled('menu_link_content', 'menu_link_content')) {
      return;
    }

    // Checking the entity_attributes field exists in the form.
    if (!isset($form['entity_attributes'])) {
      return;
    }

    // Do not make any adjustments for users without edit permission.
    if (!$this->permissions->hasPermission('menu_link_content', 'menu_link_content')) {
      return;
    }

    // Adding the details wrapper here because the one that added by the widget
    // is placed inside an additional container, and it doesn't look good.
    $form['entity_attributes']['#type'] = 'details';
    $form['entity_attributes']['#group'] = 'advanced';
    $form['entity_attributes']['#title'] = $this->t('Attributes');
    $form['entity_attributes']['#open'] = FALSE;
  }

  /**
   * Adds the entity attributes field to the Static Menu Link form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $form_id
   *   The form ID.
   */
  public function alterStaticMenuLinkForm(array &$form, FormStateInterface $form_state, string $form_id): void {
    if (!$this->pluginManager->isPluginEnabled('menu_link', 'default')) {
      return;
    }

    // Do not add the attributes field for users without edit permission.
    if (!$this->permissions->hasPermission('menu_link', 'default')) {
      return;
    }

    // Check if this is a static menu link form (MenuLinkDefaultForm).
    if (empty($form['#plugin_form'])) {
      return;
    }

    // Get the menu link plugin from the route.
    $menu_link = $this->routeMatch->getParameter('menu_link_plugin');
    if (empty($menu_link)) {
      return;
    }

    // The ID from menu link plugin.
    $plugin_id = $menu_link->getPluginId();
    // Entity Attributes static menu link plugin instance.
    $plugin = $this->pluginManager->createInstance('static_menu_link');
    $attribute_sets = $plugin->getSupportedAttributeSets();
    $current_attributes = $plugin->getAttributes($plugin_id);
    $default_yaml = $this->generateDefaultYamlValue($attribute_sets, $current_attributes);
    $description = $this->generateAttributeSetsDescription($attribute_sets);

    $form['attributes'] = [
      '#type' => 'details',
      '#title' => $this->t('Attributes'),
      '#open' => FALSE,
      '#group' => 'advanced',
      '#weight' => 10,
    ];

    // Change the parents to automatically save the field value at the right
    // place of the entity configuration tree.
    $form['attributes']['entity_attributes'] = $this->createYamlTextarea(
      $this->t('Attributes (YAML)'),
      $default_yaml,
      $description,
      ['third_party_settings', 'entity_attributes', 'attributes_data'],
    );

    $form['actions']['submit']['#submit'][] = [$this, 'submitStaticMenuLinkYaml'];
  }

  /**
   * Adds the entity attributes field to the Block form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $form_id
   *   The form ID.
   */
  public function alterBlockForm(array &$form, FormStateInterface $form_state, string $form_id): void {
    /** @var \Drupal\Core\Entity\EntityFormInterface $form_object */
    $form_object = $form_state->getFormObject();
    /** @var \Drupal\block\Entity\Block $block */
    $block = $form_object->getEntity();

    if (!$this->pluginManager->isPluginEnabled('block', 'block')) {
      return;
    }

    // Do not add the attributes field for users without edit permission.
    if (!$this->permissions->hasPermission('block', 'block')) {
      return;
    }

    // Getting the plugin and attribute sets.
    $plugin = $this->pluginManager->createInstance('block');
    $attribute_sets = $plugin->getSupportedAttributeSets();
    $default_yaml = $this->getConfigEntityYamlValue($block, $attribute_sets);
    $description = $this->generateAttributeSetsDescription($attribute_sets);

    // Create a fieldset in a settings group to place the field
    // at the appropriate place on the form.
    $form['settings']['attributes'] = [
      '#type' => 'details',
      '#title' => $this->t('Attributes'),
      '#open' => FALSE,
    ];

    // Change the parents to automatically save the field value at the right
    // place of the entity configuration tree.
    $form['settings']['attributes']['entity_attributes'] = $this->createYamlTextarea(
      $this->t('Attributes (YAML)'),
      $default_yaml,
      $description,
      ['third_party_settings', 'entity_attributes', 'attributes_data'],
    );

    // The submit handler to convert YAML to array.
    // It should run before the entity save handler.
    \array_unshift($form['actions']['submit']['#submit'], [$this, 'submitConfigEntityYaml']);
  }

  /**
   * Adds the entity attributes field to the Menu form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $form_id
   *   The form ID.
   */
  public function alterMenuForm(array &$form, FormStateInterface $form_state, string $form_id): void {
    /** @var \Drupal\Core\Entity\EntityFormInterface $form_object */
    $form_object = $form_state->getFormObject();
    /** @var \Drupal\system\Entity\Menu $menu */
    $menu = $form_object->getEntity();

    if (!$this->pluginManager->isPluginEnabled('menu', 'menu')) {
      return;
    }

    // Do not add the attributes field for users without edit permission.
    if (!$this->permissions->hasPermission('menu', 'menu')) {
      return;
    }

    // Getting the plugin and attribute sets.
    $plugin = $this->pluginManager->createInstance('menu');
    $attribute_sets = $plugin->getSupportedAttributeSets();
    $default_yaml = $this->getConfigEntityYamlValue($menu, $attribute_sets);
    $description = $this->generateAttributeSetsDescription($attribute_sets);

    // Using weight as the most simple way to place the field at the right
    // place on the form (above the links). By default, menu form items don't
    // have weight, but if they were overridden, we respect it.
    $form['links']['#weight'] ??= 2;

    $form['attributes'] = [
      '#type' => 'details',
      '#title' => $this->t('Attributes'),
      '#open' => FALSE,
      '#weight' => $form['links']['#weight'] - 1,
    ];

    $form['attributes']['entity_attributes'] = $this->createYamlTextarea(
      $this->t('Attributes (YAML)'),
      $default_yaml,
      $description,
      ['third_party_settings', 'entity_attributes', 'attributes_data']
    );

    // The submit handler to convert YAML to array.
    // It should run before the entity save handler.
    \array_unshift($form['actions']['submit']['#submit'], [$this, 'submitConfigEntityYaml']);
  }

}
