<?php

namespace Drupal\entity_attributes;

use Drupal\Component\Plugin\PluginInspectionInterface;

/**
 * Defines an interface for Entity Attributes plugins.
 */
interface EntityAttributesInterface extends PluginInspectionInterface {

  /**
   * Get entity types this plugin supports.
   *
   * @return array
   *   Array of entity type IDs this plugin supports.
   */
  public function getEntityTypes(): array;

  /**
   * Get label for the given entity type.
   *
   * @param string $entity_type
   *   The entity type ID.
   *
   * @return string
   *   The entity type label.
   */
  public function getEntityTypeLabel(string $entity_type): string;

  /**
   * Get all available bundles for this entity type.
   *
   * @param string $entity_type
   *   The entity type ID.
   *
   * @return array
   *   An array of bundle labels keyed by the machine name.
   */
  public function getBundles(string $entity_type): array;

  /**
   * Add the attributes field to a specific bundle.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return bool
   *   TRUE if the field was added successfully, FALSE otherwise.
   */
  public function addAttributesField(string $entity_type, string $bundle): bool;

  /**
   * Remove the attributes field from a specific bundle.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return bool
   *   TRUE if the field was removed successfully, FALSE otherwise.
   */
  public function removeAttributesField(string $entity_type, string $bundle): bool;

  /**
   * Check if the attributes field exists for a specific bundle.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return bool
   *   TRUE if the field exists, FALSE otherwise.
   */
  public function hasAttributesField(string $entity_type, string $bundle): bool;

  /**
   * Update field storage lock status.
   *
   * @param string $entity_type
   *   The entity type ID.
   *
   * @return void
   *   The nothing.
   */
  public function updateFieldStorageLockStatus(string $entity_type): void;

  /**
   * Get the field name used for attributes in this entity type.
   *
   * @return string
   *   The field name.
   */
  public function getFieldName(): string;

  /**
   * Get the attribute sets supported by the entity template.
   *
   * @return array
   *   The attribute sets supported by the entity.
   */
  public function getSupportedAttributeSets(): array;

}
