<?php

namespace Drupal\entity_attributes;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\entity_attributes\Attribute\EntityAttributes as EntityAttributesAttribute;

/**
 * Provides the Entity Attributes plugin manager.
 */
class EntityAttributesManager extends DefaultPluginManager {

  /**
   * Constructs a new EntityAttributesManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler to invoke the alter hook with.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct(
      'Plugin/EntityAttributes',
      $namespaces,
      $module_handler,
      EntityAttributesInterface::class,
      EntityAttributesAttribute::class
    );

    $this->alterInfo('entity_attributes_info');
    $this->setCacheBackend($cache_backend, 'entity_attributes_plugins');
  }

  /**
   * Get a plugin for a specific entity type.
   *
   * @param string $entity_type
   *   The entity type ID.
   *
   * @return \Drupal\entity_attributes\EntityAttributesInterface|null
   *   The plugin instance or NULL if not found.
   */
  public function getPluginForEntityType(string $entity_type): ?EntityAttributesInterface {
    $definitions = $this->getDefinitions();
    foreach ($definitions as $plugin_id => $definition) {
      // Check if the plugin supports this entity type (single or multiple).
      if ($this->pluginSupportsEntityType($definition, $entity_type)) {
        /** @var \Drupal\entity_attributes\EntityAttributesInterface $plugin */
        $plugin = $this->createInstance($plugin_id);

        return $plugin;
      }
    }

    return NULL;
  }

  /**
   * Check if a plugin definition supports a given entity type.
   *
   * @param array $definition
   *   The plugin definition.
   * @param string $entity_type
   *   The entity type to check.
   *
   * @return bool
   *   TRUE if the plugin supports the entity type.
   */
  protected function pluginSupportsEntityType(array $definition, string $entity_type): bool {
    // Check static entity types first.
    if (!empty($definition['entity_types'])) {
      return \in_array($entity_type, $definition['entity_types']);
    }

    // For plugins without static entity type definition,
    // check dynamic getEntityTypes() method.
    $plugin = $this->createInstance($definition['id']);

    return \in_array($entity_type, $plugin->getEntityTypes());
  }

  /**
   * Check if a plugin is enabled for a specific entity type and bundle.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param string $bundle
   *   The bundle ID.
   *
   * @return bool
   *   TRUE if the plugin is enabled for the given entity type and bundle.
   */
  public function isPluginEnabled(string $entity_type, string $bundle): bool {
    $plugin = $this->getPluginForEntityType($entity_type);
    if (!$plugin) {
      return FALSE;
    }

    return $plugin->hasAttributesField($entity_type, $bundle);
  }

}
