<?php

declare(strict_types=1);

namespace Drupal\entity_attributes;

use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Provides dynamic permissions for entity attributes plugins.
 */
class EntityAttributesPermissions {

  use StringTranslationTrait;

  /**
   * The entity attributes plugin manager.
   */
  protected EntityAttributesManager $pluginManager;

  /**
   * The current user.
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Constructs an EntityAttributesPermissions object.
   *
   * @param \Drupal\entity_attributes\EntityAttributesManager $plugin_manager
   *   The entity attributes plugin manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(EntityAttributesManager $plugin_manager, AccountProxyInterface $current_user) {
    $this->pluginManager = $plugin_manager;
    $this->currentUser = $current_user;
  }

  /**
   * Returns dynamic permissions for entity attributes plugins.
   *
   * @return array
   *   The permissions array.
   */
  public function permissions(): array {
    $permissions = [];

    $definitions = $this->pluginManager->getDefinitions();
    foreach ($definitions as $plugin_id => $definition) {
      try {
        $plugin = $this->pluginManager->createInstance($plugin_id);
        $entity_type_ids = $plugin->getEntityTypes();

        foreach ($entity_type_ids as $entity_type_id) {
          $bundles = $plugin->getBundles($entity_type_id);
          $entity_type_label = $plugin->getEntityTypeLabel($entity_type_id);

          foreach ($bundles as $bundle_id => $bundle_label) {
            $permission_name = "edit entity attributes {$entity_type_id} {$bundle_id}";

            $permissions[$permission_name] = [
              'title' => $this->t('%entity_type - %bundle: Edit entity attributes', [
                '%entity_type' => $entity_type_label,
                '%bundle' => $bundle_label,
              ]),
            ];
          }
        }
      }
      catch (\Exception $e) {
        // Skip plugins that can't be instantiated.
        continue;
      }
    }

    return $permissions;
  }

  /**
   * Check if the current user has permission to edit attributes for an entity.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param string $bundle
   *   The bundle ID.
   *
   * @return bool
   *   TRUE if the user has permission, FALSE otherwise.
   */
  public function hasPermission(string $entity_type, string $bundle): bool {
    $permission_name = "edit entity attributes {$entity_type} {$bundle}";
    return $this->currentUser->hasPermission($permission_name);
  }

}
