<?php

namespace Drupal\entity_attributes;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Template\Attribute;

/**
 * Service for processing entity attributes in preprocessing hooks.
 */
class EntityAttributesProcessor {

  /**
   * The entity attributes plugin manager.
   */
  protected EntityAttributesManager $pluginManager;

  /**
   * The logger factory.
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs an EntityAttributesProcessor object.
   *
   * @param \Drupal\entity_attributes\EntityAttributesManager $plugin_manager
   *   The entity attributes plugin manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    EntityAttributesManager $plugin_manager,
    LoggerChannelFactoryInterface $logger_factory,
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    $this->pluginManager = $plugin_manager;
    $this->loggerFactory = $logger_factory;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Processes attributes for content entities.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity being processed.
   * @param array $variables
   *   The template variables array.
   * @param string $entity_type
   *   The entity type ID (e.g., 'node', 'taxonomy_term').
   */
  public function processContentEntityAttributes(FieldableEntityInterface $entity, array &$variables, string $entity_type): void {
    if (!$this->pluginManager->isPluginEnabled($entity_type, $entity->bundle())) {
      return;
    }

    $plugin = $this->pluginManager->getPluginForEntityType($entity_type);
    if (!$plugin) {
      return;
    }

    $attributes_field_name = $plugin->getFieldName();
    if (!$entity->hasField($attributes_field_name)) {
      return;
    }

    if ($entity->get($attributes_field_name)->isEmpty()) {
      return;
    }

    $attributes_data_string = $entity->get($attributes_field_name)->getString();
    $attributes_data = Yaml::decode($attributes_data_string);
    $this->applyAttributesToVariables($attributes_data, $variables, $plugin->getSupportedAttributeSets());
  }

  /**
   * Processes attributes for config entities.
   *
   * @param \Drupal\Core\Config\Entity\ConfigEntityInterface $entity
   *   The entity being processed.
   * @param array $variables
   *   The template variables array.
   * @param string $entity_type_id
   *   The entity type ID (e.g., 'block').
   */
  public function processConfigEntityAttributes(ConfigEntityInterface $entity, array &$variables, string $entity_type_id): void {
    $bundle = $entity->bundle() ?: $entity_type_id;
    if (!$this->pluginManager->isPluginEnabled($entity_type_id, $bundle)) {
      return;
    }

    $plugin = $this->pluginManager->createInstance($entity_type_id);
    $attributes_data = $entity->getThirdPartySetting('entity_attributes', 'attributes_data');
    if (empty($attributes_data)) {
      return;
    }

    $this->applyAttributesToVariables($attributes_data, $variables, $plugin->getSupportedAttributeSets());
  }

  /**
   * Recursively process menu items to add entity attributes.
   *
   * @param array $items
   *   The menu items array.
   */
  public function processMenuItems(array &$items): void {
    foreach ($items as &$item) {
      // Check if this menu item has an entity (menu_link_content).
      if (isset($item['original_link']) && \str_starts_with($item['original_link']->getPluginId(), 'menu_link_content:')) {
        $this->processMenuLinkContentAttributes($item);
      }
      // Handle static menu links.
      elseif (isset($item['original_link'])) {
        $this->processStaticMenuLinkAttributes($item['original_link']->getPluginId(), $item);
      }

      if (empty($item['below'])) {
        continue;
      }

      // Process child items recursively.
      $this->processMenuItems($item['below']);
    }
  }

  /**
   * Process attributes for content menu links.
   *
   * @param array $item
   *   The menu item array.
   */
  protected function processMenuLinkContentAttributes(array &$item): void {
    if (!$this->pluginManager->isPluginEnabled('menu_link_content', 'menu_link_content')) {
      return;
    }

    $entity_id = $item['original_link']->getMetaData()['entity_id'] ?? NULL;
    if (empty($entity_id)) {
      return;
    }

    $menu_link_content = $this->entityTypeManager->getStorage('menu_link_content')->load($entity_id);
    if (empty($menu_link_content)) {
      return;
    }

    $this->processContentEntityAttributes($menu_link_content, $item, 'menu_link_content');
  }

  /**
   * Process attributes for static menu links.
   *
   * @param string $plugin_id
   *   The menu link plugin ID.
   * @param array $item
   *   The menu item array.
   */
  protected function processStaticMenuLinkAttributes(string $plugin_id, array &$item): void {
    if (!$this->pluginManager->isPluginEnabled('menu_link', 'default')) {
      return;
    }

    $plugin = $this->pluginManager->createInstance('static_menu_link');
    $attributes = $plugin->getAttributes($plugin_id);

    if (empty($attributes)) {
      return;
    }

    $this->applyAttributesToVariables($attributes, $item, $plugin->getSupportedAttributeSets());
  }

  /**
   * Applies attribute data to template variables.
   *
   * @param array $attributes_data
   *   The decoded attributes data.
   * @param array $variables
   *   The template variables array (passed by reference).
   * @param array $attribute_sets
   *   The supported attribute sets for this entity type.
   */
  protected function applyAttributesToVariables(array $attributes_data, array &$variables, array $attribute_sets): void {
    foreach ($attribute_sets as $set_name) {
      if (empty($attributes_data[$set_name])) {
        continue;
      }

      // Create Attribute object from field data.
      $field_attributes = new Attribute($attributes_data[$set_name]);
      // Merge with existing attributes, giving precedence to field values.
      if (isset($variables[$set_name])) {
        // Convert to Attribute object if it's an array.
        if (\is_array($variables[$set_name])) {
          $variables[$set_name] = new Attribute($variables[$set_name]);
        }
        // Merge the field attributes.
        $variables[$set_name] = $variables[$set_name]->merge($field_attributes);
      }
      else {
        $variables[$set_name] = $field_attributes;
      }
    }
  }

}
