<?php

declare(strict_types=1);

namespace Drupal\entity_attributes;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleUninstallValidatorInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Url;

/**
 * Prevents uninstalling Entity Attributes when plugins are enabled.
 */
class EntityAttributesUninstallValidator implements ModuleUninstallValidatorInterface {

  use StringTranslationTrait;

  /**
   * The entity attributes plugin manager.
   */
  protected EntityAttributesManager $pluginManager;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs an EntityAttributesUninstallValidator object.
   *
   * @param \Drupal\entity_attributes\EntityAttributesManager $plugin_manager
   *   The entity attributes plugin manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   */
  public function __construct(
    EntityAttributesManager $plugin_manager,
    ConfigFactoryInterface $config_factory,
    TranslationInterface $string_translation,
  ) {
    $this->pluginManager = $plugin_manager;
    $this->configFactory = $config_factory;
    $this->stringTranslation = $string_translation;
  }

  /**
   * {@inheritdoc}
   */
  public function validate($module) {
    $reasons = [];

    if ($module !== 'entity_attributes') {
      return $reasons;
    }

    // Check if any plugins are enabled.
    $config = $this->configFactory->get('entity_attributes.settings');
    $enabled_bundles = $config->get('enabled_bundles') ?: [];
    $enabled_plugins = [];
    foreach ($enabled_bundles as $entity_type => $bundles) {
      if (!empty($bundles)) {
        $plugin = $this->pluginManager->getPluginForEntityType($entity_type);
        if ($plugin) {
          $plugin_definition = $this->pluginManager->getDefinition($plugin->getPluginId());
          $enabled_plugins[] = $plugin_definition['label'];
        }
      }
    }

    if (!empty($enabled_plugins)) {
      $config_url = Url::fromRoute('entity_attributes.config_form');
      $plugin_list = \implode(', ', $enabled_plugins);

      $reasons[] = $this->t(
        'The following Entity Attributes plugins are enabled: %plugins. <a href="@url">Disable all plugins</a> before uninstalling.',
        [
          '%plugins' => $plugin_list,
          '@url' => $config_url->toString(),
        ]
      );
    }

    return $reasons;
  }

}
