<?php

namespace Drupal\entity_attributes\Form;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\entity_attributes\EntityAttributesManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a settings form for Entity Attributes.
 */
class EntityAttributesConfigForm extends ConfigFormBase {

  /**
   * The entity attributes plugin manager.
   */
  protected EntityAttributesManager $entityAttributesManager;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity field manager.
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->entityAttributesManager = $container->get('plugin.manager.entity_attributes');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->entityFieldManager = $container->get('entity_field.manager');
    $instance->moduleHandler = $container->get('module_handler');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'entity_attributes_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['entity_attributes.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // Check if this is a confirmation step.
    $confirm_step = $form_state->get('confirm_step');
    if ($confirm_step) {
      return $this->buildConfirmForm($form, $form_state);
    }

    $form['description'] = [
      '#markup' => '<p>' . $this->t('Configure which entity types and bundles should have the Attributes field on the edit form.') . '</p>',
    ];

    $form['entity_types'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Entity Types'),
    ];

    // Group plugins by tab.
    $definitions = $this->entityAttributesManager->getDefinitions();
    $tabs = [];

    foreach ($definitions as $plugin_id => $definition) {
      $plugin = $this->entityAttributesManager->createInstance($plugin_id);
      $entity_types = $plugin->getEntityTypes();

      // Process each entity type this plugin supports.
      foreach ($entity_types as $entity_type) {
        $bundles = $plugin->getBundles($entity_type);

        if (empty($bundles)) {
          continue;
        }

        // Get the config tab and section from the plugin definition.
        $config_tab = $definition['config_tab'] ?? 'other';
        $config_section = $definition['config_section'] ?? 'main';

        if (!isset($tabs[$config_tab])) {
          $tabs[$config_tab] = [
            'main' => [],
            'bundles' => [],
            'plugins' => [],
          ];
        }

        $tabs[$config_tab][$config_section][] = [
          'plugin_id' => $plugin_id,
          'plugin' => $plugin,
          'definition' => $definition,
          'entity_type' => $entity_type,
          'bundles' => $bundles,
        ];
      }
    }

    // Build tabs.
    foreach ($tabs as $tab_name => $sections) {
      // Skip empty tabs.
      if (empty($sections['main']) && empty($sections['bundles']) && empty($sections['plugins'])) {
        continue;
      }

      // Find the tab label from plugin definitions.
      $tab_label = NULL;
      foreach ($sections as $section_plugins) {
        foreach ($section_plugins as $plugin_data) {
          $definition = $plugin_data['definition'];
          if (isset($definition['config_tab_label'])) {
            $tab_label = $definition['config_tab_label'];
            break 2;
          }
        }
      }

      // Fall back to the translatable tab name if no plugin-defined label found.
      if (!$tab_label) {
        $fallback_labels = [
          'node' => $this->t('Node'),
          'taxonomy' => $this->t('Taxonomy'),
          'block' => $this->t('Block'),
          'menu' => $this->t('Menu'),
          'other' => $this->t('Other'),
        ];
        $tab_label = $fallback_labels[$tab_name] ?? $this->t('Other');
      }

      $form['tab_' . $tab_name] = [
        '#type' => 'details',
        '#title' => $tab_label,
        '#group' => 'entity_types',
      ];

      // Build the main section (untitled).
      if (!empty($sections['main'])) {
        $this->buildSection($form, $form_state, 'tab_' . $tab_name, $sections['main']);
      }

      // Build bundles section.
      if (!empty($sections['bundles'])) {
        $this->buildSection($form, $form_state, 'tab_' . $tab_name, $sections['bundles'], $this->t('Bundles'));
      }

      // Build plugins section.
      if (!empty($sections['plugins'])) {
        $this->buildSection($form, $form_state, 'tab_' . $tab_name, $sections['plugins'], $this->t('Plugins'));
      }
    }

    // CodeMirror editor settings.
    $config = $this->config('entity_attributes.settings');
    $codemirror_enabled = $this->moduleHandler->moduleExists('codemirror_editor');

    $form['use_codemirror'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use CodeMirror editor for Attributes field'),
      '#default_value' => $config->get('use_codemirror') ?? FALSE,
      '#disabled' => !$codemirror_enabled,
      '#description' => $codemirror_enabled
        ? $this->t('Enable syntax highlighting and advanced editing features for YAML attributes.')
        : $this->t('The CodeMirror Editor module must be enabled to use this feature.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Build a section of the form for a group of plugins.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $parent_key
   *   The parent key in the form array.
   * @param array $plugins
   *   The plugins in this section.
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup|null $section_label
   *   The section label (optional).
   */
  protected function buildSection(array &$form, FormStateInterface $form_state, string $parent_key, array $plugins, ?TranslatableMarkup $section_label = NULL): void {
    // Combine all plugins in this section into a single checkboxes list.
    $all_options = [];
    $all_default_values = [];

    foreach ($plugins as $plugin_data) {
      $plugin = $plugin_data['plugin'];
      $bundles = $plugin_data['bundles'];
      $entity_type = $plugin_data['entity_type'];

      // Add bundles to the combined options and default values.
      foreach ($bundles as $bundle_id => $bundle_label) {
        $all_options['bundles_' . $entity_type . '[' . $bundle_id . ']'] = $bundle_label;
        if ($plugin->hasAttributesField($entity_type, $bundle_id)) {
          $all_default_values[] = 'bundles_' . $entity_type . '[' . $bundle_id . ']';
        }
      }
    }

    // Navigate to the correct form element.
    $form_element = &$form;
    foreach (\explode('][', $parent_key) as $key) {
      if (!isset($form_element[$key])) {
        $form_element[$key] = [];
      }
      $form_element = &$form_element[$key];
    }

    // Create a single checkboxes element for the entire section.
    $section_key = 'section_' . \md5($parent_key . ($section_label ? $section_label->__toString() : ''));
    $form_element[$section_key] = [
      '#type' => 'checkboxes',
      '#options' => $all_options,
      '#default_value' => $all_default_values,
    ];

    // Add a section label if provided.
    if ($section_label !== NULL) {
      $form_element[$section_key]['#title'] = $section_label;
    }
  }

  /**
   * Build the confirmation form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  protected function buildConfirmForm(array $form, FormStateInterface $form_state): array {
    $bundles_to_remove = $form_state->get('bundles_to_remove');

    $form['#title'] = $this->t('Confirm field removal');

    $form['warning'] = [
      '#markup' => '<div class="messages messages--warning">' .
      $this->t('<strong>Warning:</strong> You are about to remove Entity Attributes fields from the following bundles. All data stored in these fields will be permanently lost.') .
      '</div>',
    ];

    $items = [];
    foreach ($bundles_to_remove as $entity_type => $bundles) {
      $definitions = $this->entityAttributesManager->getDefinitions();
      foreach ($definitions as $plugin_id => $definition) {
        $plugin = $this->entityAttributesManager->createInstance($plugin_id);
        $entity_types = $plugin->getEntityTypes();

        if (\in_array($entity_type, $entity_types)) {
          $bundle_options = $plugin->getBundles($entity_type);
          $entity_type_label = $plugin->getEntityTypeLabel($entity_type);

          foreach ($bundles as $bundle_id) {
            if (isset($bundle_options[$bundle_id])) {
              $items[] = $this->t('@entity_type - @bundle', [
                '@entity_type' => $entity_type_label,
                '@bundle' => $bundle_options[$bundle_id],
              ]);
            }
          }
        }
      }
    }

    $form['bundles_list'] = [
      '#theme' => 'item_list',
      '#items' => $items,
      '#title' => $this->t('Fields will be removed from:'),
    ];

    $form['confirm_text'] = [
      '#markup' => '<p>' . $this->t('This action cannot be undone. Are you sure you want to proceed?') . '</p>',
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['confirm'] = [
      '#type' => 'submit',
      '#value' => $this->t('Yes, remove fields'),
      '#button_type' => 'primary',
      '#submit' => ['::confirmSubmit'],
    ];

    $form['actions']['cancel'] = [
      '#type' => 'submit',
      '#value' => $this->t('Cancel'),
      '#submit' => ['::cancelSubmit'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $enabled_bundles = [];
    $bundles_to_remove = [];
    $has_removals = FALSE;

    // Get all form values and process combined checkboxes structure.
    $values = $form_state->getValues();

    // Extract enabled bundles from section checkboxes.
    foreach ($values as $key => $value) {
      if (\str_starts_with($key, 'section_') && \is_array($value)) {
        foreach ($value as $checkbox_key => $checkbox_value) {
          if ($checkbox_value && \preg_match('/^bundles_(\w+)\[(\w+)\]$/', $checkbox_key, $matches)) {
            $entity_type = $matches[1];
            $bundle_id = $matches[2];
            $enabled_bundles[$entity_type][] = $bundle_id;
          }
        }
      }
    }

    $definitions = $this->entityAttributesManager->getDefinitions();
    foreach ($definitions as $plugin_id => $definition) {
      $plugin = $this->entityAttributesManager->createInstance($plugin_id);
      $entity_types = $plugin->getEntityTypes();

      // Process each entity type this plugin supports.
      foreach ($entity_types as $entity_type) {
        $bundles = $plugin->getBundles($entity_type);
        if (empty($bundles)) {
          continue;
        }

        $entity_enabled_bundles = $enabled_bundles[$entity_type] ?? [];
        // Check for bundles that need field removal.
        foreach ($bundles as $bundle_id => $bundle_label) {
          $should_have_field = \in_array($bundle_id, $entity_enabled_bundles);
          $currently_has_field = $plugin->hasAttributesField($entity_type, $bundle_id);

          if (!$should_have_field && $currently_has_field) {
            $bundles_to_remove[$entity_type][] = $bundle_id;
            $has_removals = TRUE;
          }
        }
      }
    }

    // If there are fields to remove, show a confirmation page.
    if ($has_removals) {
      $form_state->set('bundles_to_remove', $bundles_to_remove);
      $form_state->set('enabled_bundles', $enabled_bundles);
      $form_state->set('confirm_step', TRUE);
      $form_state->setRebuild(TRUE);
      return;
    }

    // Save the CodeMirror setting.
    $config = $this->config('entity_attributes.settings');
    $config->set('use_codemirror', $form_state->getValue('use_codemirror'));
    $config->save();

    // Process field additions only (no removals).
    $this->processFieldChanges($enabled_bundles, $form_state);
    parent::submitForm($form, $form_state);
  }

  /**
   * Process field changes (additions and removals).
   */
  protected function processFieldChanges(array $enabled_bundles, FormStateInterface $form_state): void {
    $config = $this->config('entity_attributes.settings');
    $messages = [];

    $definitions = $this->entityAttributesManager->getDefinitions();
    foreach ($definitions as $plugin_id => $definition) {
      $plugin = $this->entityAttributesManager->createInstance($plugin_id);
      $entity_types = $plugin->getEntityTypes();

      // Process each entity type this plugin supports.
      foreach ($entity_types as $entity_type) {
        $bundles = $plugin->getBundles($entity_type);
        if (empty($bundles)) {
          continue;
        }

        $entity_type_label = $plugin->getEntityTypeLabel($entity_type);
        $entity_enabled_bundles = $enabled_bundles[$entity_type] ?? [];
        // Process each bundle.
        foreach ($bundles as $bundle_id => $bundle_label) {
          $should_have_field = \in_array($bundle_id, $entity_enabled_bundles);
          $currently_has_field = $plugin->hasAttributesField($entity_type, $bundle_id);

          if ($should_have_field && !$currently_has_field) {
            // Add field.
            if ($plugin->addAttributesField($entity_type, $bundle_id)) {
              $messages['success'][] = $this->t('Added %field_name field to %entity_type - %bundle.', [
                '%field_name' => $this->t('Attributes'),
                '%entity_type' => $entity_type_label,
                '%bundle' => $bundle_label,
              ]);
            }
            else {
              $messages['error'][] = $this->t('Failed to add %field_name field to %entity_type - %bundle.', [
                '%field_name' => $this->t('Attributes'),
                '%entity_type' => $entity_type_label,
                '%bundle' => $bundle_label,
              ]);
            }
          }
          elseif (!$should_have_field && $currently_has_field) {
            // Remove the field.
            if ($plugin->removeAttributesField($entity_type, $bundle_id)) {
              $messages['success'][] = $this->t('Removed %field_name field from %entity_type - %bundle.', [
                '%field_name' => $this->t('Attributes'),
                '%entity_type' => $entity_type_label,
                '%bundle' => $bundle_label,
              ]);
            }
            else {
              $messages['error'][] = $this->t('Failed to remove %field_name field from %entity_type - %bundle.', [
                '%field_name' => $this->t('Attributes'),
                '%entity_type' => $entity_type_label,
                '%bundle' => $bundle_label,
              ]);
            }
          }
        }
      }
    }

    // Save configuration AFTER field operations are complete.
    $config->set('enabled_bundles', $enabled_bundles);
    $config->save();

    // Clear caches to ensure field definitions are updated.
    $this->entityTypeManager->clearCachedDefinitions();
    $this->entityFieldManager->clearCachedFieldDefinitions();

    // Update field storage lock status for all entity types.
    foreach ($definitions as $plugin_id => $definition) {
      $plugin = $this->entityAttributesManager->createInstance($plugin_id);
      $entity_types = $plugin->getEntityTypes();

      // Process each entity type this plugin supports.
      foreach ($entity_types as $entity_type) {
        $plugin->updateFieldStorageLockStatus($entity_type);
      }
    }

    // Display all messages with appropriate types.
    foreach ($messages as $type => $type_messages) {
      foreach ($type_messages as $message) {
        if ($type === 'error') {
          $this->messenger()->addError($message);
        }
        else {
          $this->messenger()->addMessage($message);
        }
      }
    }
  }

  /**
   * Submit a handler for the confirmation form.
   */
  public function confirmSubmit(array &$form, FormStateInterface $form_state): void {
    $enabled_bundles = $form_state->get('enabled_bundles');
    $this->processFieldChanges($enabled_bundles, $form_state);

    // Clear the form state to prevent rebuild.
    $form_state->set('confirm_step', FALSE);
    $form_state->set('bundles_to_remove', NULL);
    $form_state->set('enabled_bundles', NULL);

    parent::submitForm($form, $form_state);
  }

  /**
   * Submit handler for the cancel action.
   */
  public function cancelSubmit(array &$form, FormStateInterface $form_state): void {
    // Clear the form state and rebuild to show the main form.
    $form_state->set('confirm_step', FALSE);
    $form_state->set('bundles_to_remove', NULL);
    $form_state->set('enabled_bundles', NULL);
    $form_state->setRebuild(TRUE);

    $this->messenger()->addMessage($this->t('Field removal cancelled.'));
  }

}
