<?php

declare(strict_types=1);

namespace Drupal\entity_attributes\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\entity_attributes\EntityAttributesField;

/**
 * Hook implementations for entity attributes field handling.
 */
class FieldHooks {

  /**
   * The entity attributes field handler.
   */
  protected EntityAttributesField $fieldHandler;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs an EntityAttributesHooks object.
   *
   * @param \Drupal\entity_attributes\EntityAttributesField $field_handler
   *   The entity attributes field handler.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    EntityAttributesField $field_handler,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->fieldHandler = $field_handler;
    $this->configFactory = $config_factory;
  }

  /**
   * Implements hook_field_widget_single_element_form_alter().
   *
   * @usedby NodeAttributes
   * @usedby TaxonomyTermAttributes
   * @usedby MenuLinkContentAttributes
   */
  #[Hook('field_widget_single_element_form_alter')]
  public function fieldWidgetSingleElementFormAlter(array &$element, FormStateInterface $form_state, array $context): void {
    // Add enhancements to the entity_attributes fields for content entities.
    $this->fieldHandler->alterFieldWidgetSingleElementForm($element, $form_state, $context);
  }

  /**
   * Implements hook_entity_bundle_field_info_alter().
   *
   * @usedby NodeAttributes
   * @usedby TaxonomyTermAttributes
   * @usedby MenuLinkContentAttributes
   */
  #[Hook('entity_bundle_field_info_alter')]
  public function entityBundleFieldInfoAlter(array &$fields, EntityTypeInterface $entity_type, string $bundle): void {
    // Add validation constraint to the entity_attributes fields
    // for content entities.
    if (isset($fields['entity_attributes'])) {
      // For field items, we need to apply the constraint to
      // the 'value' property.
      $fields['entity_attributes']->addPropertyConstraints('value', [
        'ValidYaml' => [],
      ]);
    }
  }

  /**
   * Implements hook_entity_type_build().
   *
   * @usedby MenuLinkContentAttributes
   */
  #[Hook('entity_type_build')]
  public function entityTypeBuild(array &$entity_types): void {
    if (empty($entity_types['menu_link_content'])) {
      return;
    }

    // Cannot use standard $pluginManager->isPluginEnabled() method here,
    // since for the config entities it is based on the field configuration,
    // but in this case it is not available yet.
    $config = $this->configFactory->get('entity_attributes.settings');
    $enabled_bundles = $config->get('enabled_bundles.menu_link_content') ?: [];
    if (!\in_array('menu_link_content', $enabled_bundles)) {
      return;
    }

    /** @var \Drupal\Core\Entity\EntityTypeInterface $entity_type */
    $entity_type = &$entity_types['menu_link_content'];
    // // Make menu_link_content entity type fieldable.
    $entity_type->set('fieldable', TRUE);
    // Set the bundle key to the entity type itself (no bundles).
    $entity_type->set('bundle_keys', ['bundle' => 'bundle']);
  }

  /**
   * Implements hook_form_BASE_FORM_ID_alter() for menu_link_content_form.
   *
   * @usedby MenuLinkContentAttributes
   */
  #[Hook('form_menu_link_content_form_alter')]
  public function formMenuLinkContentFormAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    // Adjust the entity_attributes field for menu link content.
    $this->fieldHandler->alterMenuLinkContentForm($form, $form_state, $form_id);
  }

  /**
   * Implements hook_form_FORM_ID_alter() for menu_link_edit.
   *
   * @usedby StaticMenuLinkAttributes
   */
  #[Hook('form_menu_link_edit_alter')]
  public function formMenuLinkEditAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    // Add the entity_attributes field for static menu links.
    $this->fieldHandler->alterStaticMenuLinkForm($form, $form_state, $form_id);
  }

  /**
   * Implements hook_form_BASE_FORM_ID_alter() for block_form.
   *
   * @usedby BlockAttributes
   */
  #[Hook('form_block_form_alter')]
  public function formBlockFormAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    // Add the entity_attributes field for blocks.
    $this->fieldHandler->alterBlockForm($form, $form_state, $form_id);
  }

  /**
   * Implements hook_form_BASE_FORM_ID_alter() for menu_form.
   *
   * @usedby MenuAttributes
   */
  #[Hook('form_menu_form_alter')]
  public function formMenuFormAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    // Add the entity_attributes field for menus.
    $this->fieldHandler->alterMenuForm($form, $form_state, $form_id);
  }

}
