<?php

declare(strict_types=1);

namespace Drupal\entity_attributes\Hook;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\entity_attributes\EntityAttributesProcessor;

/**
 * Hook implementations for entity attributes preprocess.
 */
class PreprocessHooks {

  /**
   * The entity attributes processor.
   */
  protected EntityAttributesProcessor $processor;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs an EntityAttributesHooks object.
   *
   * @param \Drupal\entity_attributes\EntityAttributesProcessor $processor
   *   The entity attributes processor.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    EntityAttributesProcessor $processor,
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    $this->processor = $processor;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Implements hook_preprocess_node().
   *
   * @usedby NodeAttributes
   */
  #[Hook('preprocess_node')]
  public function preprocessNode(array &$variables): void {
    /** @var \Drupal\node\NodeInterface $node */
    $node = $variables['node'];
    $this->processor->processContentEntityAttributes($node, $variables, 'node');
  }

  /**
   * Implements hook_preprocess_taxonomy_term().
   *
   * @usedby TaxonomyTermAttributes
   */
  #[Hook('preprocess_taxonomy_term')]
  public function preprocessTaxonomyTerm(array &$variables): void {
    /** @var \Drupal\taxonomy\Entity\Term $term */
    $term = $variables['term'];
    $this->processor->processContentEntityAttributes($term, $variables, 'taxonomy_term');
  }

  /**
   * Implements hook_preprocess_menu().
   *
   * @usedby MenuAttributes
   * @usedby MenuLinkContentAttributes
   * @usedby StaticMenuLinkAttributes
   */
  #[Hook('preprocess_menu')]
  public function preprocessMenu(array &$variables): void {
    // Process menu link attributes.
    if (isset($variables['items'])) {
      $this->processor->processMenuItems($variables['items']);
    }

    // Process menu attributes.
    if (empty($variables['menu_name'])) {
      return;
    }

    $menu = $this->entityTypeManager->getStorage('menu')->load($variables['menu_name']);
    if (empty($menu)) {
      return;
    }

    $this->processor->processConfigEntityAttributes($menu, $variables, 'menu');
  }

  /**
   * Implements hook_preprocess_block().
   *
   * @usedby BlockAttributes
   */
  #[Hook('preprocess_block')]
  public function preprocessBlock(array &$variables): void {
    // Blocks coming from the page manager widget do not have IDs.
    if (empty($variables['elements']['#id'])) {
      return;
    }

    $block = $this->entityTypeManager->getStorage('block')->load($variables['elements']['#id']);
    if (empty($block)) {
      return;
    }

    $this->processor->processConfigEntityAttributes($block, $variables, 'block');
  }

}
