<?php

declare(strict_types=1);

namespace Drupal\entity_attributes\Plugin\EntityAttributes;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\entity_attributes\Attribute\EntityAttributes;
use Drupal\entity_attributes\ConfigEntityAttributesBase;

/**
 * Plugin for static menu link attributes.
 *
 * Static menu links are not true configuration entity but similar to.
 */
#[EntityAttributes(
  id: 'static_menu_link',
  label: new TranslatableMarkup('Static Menu Links'),
  entity_types: ['menu_link'],
  description: new TranslatableMarkup('Manages attributes for module-defined static menu links.'),
  config_tab: 'menu',
  config_tab_label: new TranslatableMarkup('Menu'),
  config_section: 'plugins',
)]
class StaticMenuLinkAttributes extends ConfigEntityAttributesBase {

  /**
   * The configuration name for storing a static menu link overrides.
   */
  protected const string CONFIG_NAME = 'entity_attributes.static_menu_link_overrides';

  /**
   * {@inheritdoc}
   */
  public function getEntityTypeLabel(string $entity_type): string {
    // Static menu link is not a true entity and has no label.
    return (string) $this->t('Config menu link');
  }

  /**
   * {@inheritdoc}
   */
  public function getBundles(string $entity_type): array {
    return ['default' => $this->t('Static Menu Link')];
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedAttributeSets(): array {
    return [
      'attributes',
      'link_attributes',
    ];
  }

  /**
   * Gets attributes for a static menu link by plugin ID.
   *
   * @param string $plugin_id
   *   The menu link plugin ID.
   *
   * @return array
   *   The attributes data.
   */
  public function getAttributes(string $plugin_id): array {
    $config = $this->configFactory->get(self::CONFIG_NAME);
    $encoded_id = $this->encodeId($plugin_id);
    return $config->get("definitions.{$encoded_id}.third_party_settings.entity_attributes.attributes_data") ?? [];
  }

  /**
   * Sets attributes for a static menu link by plugin ID.
   *
   * @param string $plugin_id
   *   The menu link plugin ID.
   * @param array $attributes
   *   The attributes data.
   */
  public function setAttributes(string $plugin_id, array $attributes): void {
    $config = $this->configFactory->getEditable(self::CONFIG_NAME);
    $encoded_id = $this->encodeId($plugin_id);

    if (empty($attributes)) {
      $config->clear("definitions.{$encoded_id}.third_party_settings.entity_attributes.attributes_data");

      // Clean up empty sections.
      $module_settings = $config->get("definitions.{$encoded_id}.third_party_settings.entity_attributes") ?? [];
      if (empty($module_settings)) {
        $config->clear("definitions.{$encoded_id}.third_party_settings.entity_attributes");
      }

      $third_party_settings = $config->get("definitions.{$encoded_id}.third_party_settings") ?? [];
      if (empty($third_party_settings)) {
        $config->clear("definitions.{$encoded_id}.third_party_settings");
      }

      $definition = $config->get("definitions.{$encoded_id}") ?? [];
      if (empty($definition)) {
        $config->clear("definitions.{$encoded_id}");
      }
    }
    else {
      $config->set("definitions.{$encoded_id}.third_party_settings.entity_attributes.attributes_data", $attributes);
    }

    $config->save();
  }

  /**
   * {@inheritdoc}
   */
  public function removeAttributesField(string $entity_type, string $bundle): bool {
    try {
      $config = $this->configFactory->getEditable(self::CONFIG_NAME);
      $definitions = $config->get('definitions') ?? [];
      $cleared_count = 0;

      foreach ($definitions as $plugin_id => $definition) {
        if (isset($definition['third_party_settings']['entity_attributes'])) {
          unset($definitions[$plugin_id]['third_party_settings']['entity_attributes']);

          // Clean up empty third_party_settings.
          if (empty($definitions[$plugin_id]['third_party_settings'])) {
            unset($definitions[$plugin_id]['third_party_settings']);
          }

          // Clean up empty plugin entries.
          if (empty($definitions[$plugin_id])) {
            unset($definitions[$plugin_id]);
          }

          $cleared_count++;
        }
      }

      if ($cleared_count > 0) {
        $config->set('definitions', $definitions)->save();
      }

      $this->logger->info('Cleaned entity_attributes settings from @count static menu links', [
        '@count' => $cleared_count,
      ]);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to clean up static menu link attributes: @error', [
        '@error' => $e->getMessage(),
      ]);

      return FALSE;
    }
  }

  /**
   * Encodes a plugin ID for safe storage as a configuration key.
   *
   * @param string $id
   *   The plugin ID.
   *
   * @return string
   *   The encoded plugin ID.
   */
  protected function encodeId(string $id): string {
    return \strtr($id, ['.' => '__', '__' => '___']);
  }

}
