<?php

namespace Drupal\entity_attributes\Plugin\Field\FieldWidget;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'entity_attributes_yaml' widget.
 */
#[FieldWidget(
  id: 'entity_attributes_yaml',
  label: new TranslatableMarkup('Attributes (YAML)'),
  field_types: [
    'string_long',
  ],
)]
class AttributesWidget extends WidgetBase implements ContainerFactoryPluginInterface {

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->moduleHandler = $container->get('module_handler');
    $instance->configFactory = $container->get('config.factory');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return [
      'collapsed' => TRUE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $element = [];

    $element['collapsed'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Collapsed by default'),
      '#default_value' => $this->getSetting('collapsed'),
      '#description' => $this->t('If checked, the attributes widget will be collapsed by default.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $summary = [];
    $summary[] = $this->t('Collapsed: @collapsed', [
      '@collapsed' => $this->getSetting('collapsed') ? $this->t('Yes') : $this->t('No'),
    ]);
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    $item = $items[$delta];
    $value = $item->getValue()['value'] ?? '';

    // Determine if the widget should be collapsed.
    $collapsed = $this->getSetting('collapsed');

    // If an advanced settings tabs-set is available
    // (normally rendered in the sidebar), place the field as a details element
    // in this tab-set.
    if (isset($form['advanced'])) {
      $element += [
        '#type' => 'details',
        '#title' => $this->t('Attributes'),
        '#open' => !$collapsed && !empty($value),
        '#group' => 'advanced',
        '#access' => $items->access('edit'),
        '#attributes' => [
          'class' => ['entity-attributes-form'],
        ],
        '#weight' => 30,
      ];
    }
    // The field structure on the Menu Link Content form is different,
    // and the details wrapper is added by alterMenuLinkContentForm().
    elseif ($form_state->getFormObject()->getFormId() == 'menu_link_content_menu_link_content_form') {
      $element += [
        '#type' => 'container',
      ];
    }
    // Otherwise, render as a regular details element.
    else {
      $element += [
        '#type' => 'details',
        '#title' => $this->t('Attributes'),
        '#open' => !$collapsed,
      ];
    }

    $config = $this->configFactory->get('entity_attributes.settings');
    $use_codemirror = $config->get('use_codemirror') && $this->moduleHandler->moduleExists('codemirror_editor');

    if ($use_codemirror) {
      $element['value'] = [
        '#type' => 'codemirror',
        '#title' => $this->t('Attributes (YAML)'),
        '#default_value' => $value,
        '#rows' => 4,
        '#codemirror' => [
          'mode' => 'text/x-yaml',
          'toolbar' => FALSE,
        ],
      ];
    }
    else {
      $element['value'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Attributes (YAML)'),
        '#default_value' => $value,
        '#rows' => 8,
      ];
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state): array {
    // No transformation needed - the value is already stored directly.
    return $values;
  }

}
