<?php

namespace Drupal\entity_browser_acquia_dam\Plugin\EntityBrowser\Widget;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\entity_browser\WidgetBase;
use Drupal\entity_browser\WidgetValidationManager;
use Drupal\media\MediaSourceManager;
use Drupal\acquia_dam\AcquiadamAuthService;
use Drupal\acquia_dam\Client\AcquiaDamClientFactory;
use Drupal\acquia_dam\MediaTypeResolver;
use Drupal\user\UserDataInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Core\Render\RendererInterface;

/**
 * Entity browser widget for Acquia DAM assets.
 *
 * Provides a widget for browsing and selecting Acquia DAM assets within
 * entity browsers. Supports filtering, sorting, pagination, and category
 * navigation.
 *
 * @EntityBrowserWidget(
 *   id = "entity_browser_acquia_dam_browser",
 *   label = @Translation("Acquia DAM Browser"),
 *   description = @Translation("Browse and select Acquia DAM assets"),
 *   auto_select = FALSE
 * )
 */
class AcquiaDam extends WidgetBase {

  /**
   * The DAM client factory.
   *
   * @var \Drupal\acquia_dam\Client\AcquiaDamClientFactory
   */
  protected $clientFactory;

  /**
   * The current user account.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $user;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The media source manager.
   *
   * @var \Drupal\media\MediaSourceManager
   */
  protected $sourceManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The user data manager.
   *
   * @var \Drupal\user\UserDataInterface
   */
  protected $userData;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The media type resolver.
   *
   * @var \Drupal\acquia_dam\MediaTypeResolver
   */
  protected $mediaTypeResolver;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The Acquia DAM auth service.
   *
   * @var \Drupal\acquia_dam\AcquiadamAuthService
   */
  protected $authService;

  /**
   * Constructs a new AcquiaDam widget.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event dispatcher.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\entity_browser\WidgetValidationManager $validation_manager
   *   The widget validation manager.
   * @param \Drupal\acquia_dam\Client\AcquiaDamClientFactory $client_factory
   *   The DAM client factory.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The current user account.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\media\MediaSourceManager $sourceManager
   *   The media source manager.
   * @param \Drupal\user\UserDataInterface $userData
   *   The user data manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config
   *   The config factory.
   * @param \Drupal\acquia_dam\MediaTypeResolver $media_type_resolver
   *   The media type resolver.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\acquia_dam\AcquiadamAuthService $auth_service
   *   The Acquia DAM auth service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EventDispatcherInterface $event_dispatcher, EntityTypeManagerInterface $entity_type_manager, EntityFieldManagerInterface $entity_field_manager, WidgetValidationManager $validation_manager, AcquiaDamClientFactory $client_factory, AccountInterface $account, LanguageManagerInterface $languageManager, ModuleHandlerInterface $moduleHandler, MediaSourceManager $sourceManager, UserDataInterface $userData, RequestStack $requestStack, ConfigFactoryInterface $config, MediaTypeResolver $media_type_resolver, RendererInterface $renderer, AcquiadamAuthService $auth_service) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $event_dispatcher, $entity_type_manager, $validation_manager);
    $this->clientFactory = $client_factory;
    $this->user = $account;
    $this->languageManager = $languageManager;
    $this->moduleHandler = $moduleHandler;
    $this->sourceManager = $sourceManager;
    $this->entityFieldManager = $entity_field_manager;
    $this->userData = $userData;
    $this->requestStack = $requestStack;
    $this->config = $config;
    $this->mediaTypeResolver = $media_type_resolver;
    $this->renderer = $renderer;
    $this->authService = $auth_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('event_dispatcher'),
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('plugin.manager.entity_browser.widget_validation'),
      $container->get('acquia_dam.client.factory'),
      $container->get('current_user'),
      $container->get('language_manager'),
      $container->get('module_handler'),
      $container->get('plugin.manager.media.source'),
      $container->get('user.data'),
      $container->get('request_stack'),
      $container->get('config.factory'),
      $container->get('acquia_dam.media_type_resolver'),
      $container->get('renderer'),
      $container->get('acquia_dam.authentication_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    // Build options for Acquia DAM media types.
    $media_type_options = [];
    $media_types = $this->entityTypeManager->getStorage('media_type')->loadMultiple();

    foreach ($media_types as $media_type) {
      $source_plugin_id = $media_type->getSource()->getPluginId();
      // Only include media types that use Acquia DAM asset source.
      if (strpos($source_plugin_id, 'acquia_dam_asset') === 0) {
        $media_type_options[$media_type->id()] = $media_type->label();
      }
    }

    if (empty($media_type_options)) {
      // No Acquia DAM media types found, show link to create one.
      $url = Url::fromRoute('entity.media_type.add_form')->toString();
      $form['media_type'] = [
        '#markup' => $this->t("You don't have media type of the Acquia DAM asset type. You should <a href='!link'>create one</a>", ['!link' => $url]),
      ];
    }
    else {
      $form['media_type'] = [
        '#type' => 'select',
        '#title' => $this->t('Media type'),
        '#default_value' => $this->configuration['media_type'],
        '#options' => $media_type_options,
      ];
    }

    // Format restriction configuration.
    $form['strict_format_restriction'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Strict format restriction'),
      '#default_value' => $this->configuration['strict_format_restriction'],
      '#description' => $this->t('If checked, format filter options will be restricted based on the selected media type. If unchecked, all format options will be available.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'media_type' => NULL,
      'submit_text' => $this->t('Select assets'),
      'strict_format_restriction' => TRUE,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function getForm(array &$original_form, FormStateInterface $form_state, array $additional_widget_parameters) {
    // Validate media type configuration.
    $media_type_storage = $this->entityTypeManager->getStorage('media_type');
    if (!$this->configuration['media_type'] || !($media_type = $media_type_storage->load($this->configuration['media_type']))) {
      return ['#markup' => $this->t('The media type is not configured correctly.')];
    }
    elseif (strpos($media_type->getSource()->getPluginId(), 'acquia_dam_asset') !== 0) {
      return ['#markup' => $this->t('The configured media type is not using the acquia_dam_asset plugin.')];
    }
    elseif ($this->uuid() != $form_state->getStorage()['entity_browser_current_widget']) {
      return [];
    }

    // Check if the user has access to Acquia DAM.
    $client = $this->clientFactory->getUserClient();
    try {
      $client->getCategories();
    }
    catch (\Exception $e) {
      // User is not authenticated or DAM is not configured.
      $return_link = Url::fromRoute('acquia_dam.user_auth', ['uid' => $this->user->id()], ['absolute' => TRUE])->toString();
      $auth_url = $this->authService->generateAuthUrl($return_link);
      if ($auth_url) {
        $auth_link = Url::fromUri($auth_url, ['attributes' => ['target' => '_blank']]);
        $message = $this->t('You are not authenticated. Please %authenticate to browse Acquia DAM assets. After successful authentication close this modal and reopen it to browse Acquia DAM assets.', [
          '%authenticate' => Link::fromTextAndUrl("Authenticate", $auth_link)->toString(),
        ]);
      }
      else {
        $message = $this->t('Acquia DAM module is not configured yet. Please contact your administrator to do so.');
        if ($this->user->hasPermission('administer site configuration')) {
          $message = $this->t('Acquia DAM module is not configured yet. Please %config it to start using Acquia DAM assets.', [
            '%config' => Link::createFromRoute($this->t('configure'), 'acquia_dam.config', [], ['attributes' => ['target' => '_blank']])->toString(),
          ]);
        }
      }

      $form['message'] = [
        '#theme' => 'asset_browser_message',
        '#message' => $message,
        '#attached' => ['library' => ['entity_browser_acquia_dam/asset_browser']],
      ];
      return $form;
    }

    // Build the main form.
    $form = parent::getForm($original_form, $form_state, $additional_widget_parameters);
    $config = $this->config->get('acquia_dam.settings');
    $form['#attributes']['id'] = 'acquia-dam-widget-form';
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';

    // Get the triggering element to determine form action.
    $trigger_elem = $form_state->getTriggeringElement();

    // Initialize current category object.
    $current_category = new \stdClass();
    $current_category->name = NULL;
    $current_category->path = NULL;
    $current_category->parts = [];

    // Get selected category from various sources
    // (form state, form values, user input).
    $selected_category_path = $form_state->get('selected_category_path');
    $form_category_value = $form_state->getValue('category');
    if (empty($selected_category_path) && !empty($form_category_value)) {
      $selected_category_path = $form_category_value;
    }

    $user_input = $form_state->getUserInput();
    $user_input_category = $user_input['category'] ?? NULL;
    if (empty($selected_category_path) && !empty($user_input_category)) {
      $selected_category_path = $user_input_category;
    }

    // Store the category in form state for consistency.
    if (!empty($selected_category_path)) {
      $form_state->set('selected_category_path', $selected_category_path);
    }

    // Build current category object if a category is selected.
    if ($selected_category_path) {
      $current_category = new \stdClass();
      $current_category->name = basename($selected_category_path);
      $current_category->path = $selected_category_path;
      $current_category->parts = explode('/', $selected_category_path);
    }

    // Initialize pagination and asset count variables.
    $page = 0;
    $num_per_page = $config->get('num_assets_per_page') ?? 20;
    $total_asset = 0;

    // Restore form state if this is not a reset action.
    if (isset($form_state->getCompleteForm()['widget']) && isset($trigger_elem) && $trigger_elem['#name'] != 'filter_sort_reset') {
      $widget = $form_state->getCompleteForm()['widget'];

      // Restore page number from form state.
      if (isset($widget['pager-container']) && is_numeric($widget['pager-container']['#page'])) {
        $page = intval($widget['pager-container']['#page']);
      }

      // Restore category information from form state.
      if (isset($widget['asset-container']) && isset($widget['asset-container']['#acquia_dam_category'])) {
        $current_category->name = $widget['asset-container']['#acquia_dam_category']['name'];
        $current_category->parts = $widget['asset-container']['#acquia_dam_category']['parts'];
        $current_category->links = $widget['asset-container']['#acquia_dam_category']['links'];
        $current_category->categories = $widget['asset-container']['#acquia_dam_category']['categories'];
      }

      // Restore current selections.
      if ($form_state->getValue('assets')) {
        $current_selections = $form_state->getValue('current_selections', []) + array_filter($form_state->getValue('assets', []));
        $form['current_selections'] = [
          '#type' => 'value',
          '#value' => $current_selections,
        ];
      }
    }

    // Set default page type.
    $page_type = "listing";

    // Handle form submissions based on triggering element.
    if (isset($trigger_elem)) {
      if ($trigger_elem['#name'] === 'acquia_dam_category') {
        // Category button clicked - update category and reset page.
        $current_category->name = $trigger_elem['#acquia_dam_category']['name'];
        $current_category->parts = $trigger_elem['#acquia_dam_category']['parts'];
        $current_category->links = $trigger_elem['#acquia_dam_category']['links'];
        $page = 0;
      }
      if ($trigger_elem['#name'] === 'breadcrumb') {
        // Breadcrumb navigation - update category path.
        $current_category->name = $trigger_elem["#category_name"];
        $current_category->parts = $trigger_elem["#parts"];
      }
      if ($trigger_elem['#name'] === 'acquia_dam_pager') {
        // Pagination - update page and maintain category.
        $page_type = $trigger_elem['#page_type'];
        $current_category->name = $trigger_elem['#current_category']->name ?? NULL;
        $current_category->parts = $trigger_elem['#current_category']->parts ?? [];
        $page = intval($trigger_elem['#acquia_dam_page']);
      }
      if ($trigger_elem['#name'] === 'filter_sort_submit') {
        // Apply filters - switch to search mode and reset page.
        $page_type = "search";
        $page = 0;
      }
      if ($trigger_elem['#name'] === 'filter_sort_reset') {
        // Reset button - clear everything and return to default state.
        $form_state->setValue('category', 'all');
        $form_state->setValue('query', '');
        $form_state->setValue('sortby', 'created_date');
        $form_state->setValue('sortdir', 'desc');

        // Reset format_type to the configured default instead of empty.
        $default_format = $this->getDefaultFormatForConfiguredMediaType();
        $form_state->setValue('format_type', $default_format);

        $form_state->set('selected_category_path', NULL);
        $form_state->setUserInput([]);
        $page = 0;

        // Reset current category to default state.
        $current_category = new \stdClass();
        $current_category->name = NULL;
        $current_category->path = NULL;
        $current_category->parts = [];

        // Force a search for all assets.
        $final_query = '';
      }
    }

    // Calculate pagination offset.
    $offset = $num_per_page * $page;

    // Get sort parameters from form state.
    $sort_by = ($form_state->getValue('sortby') ?? 'created_date');
    $sort_dir = $form_state->getValue('sortdir') ?? 'desc';
    $sort_by = ($sort_dir == 'desc') ? '-' . $sort_by : $sort_by;

    // Get filter parameters from form state.
    $format_type = $form_state->getValue('format_type');
    if ($format_type === NULL || $format_type === '') {
      // Set default format based on configured media type if not set.
      $format_type = $this->getDefaultFormatForConfiguredMediaType();
    }
    $filter_type = !empty($format_type) ? 'ft:' . $format_type : '';
    $keyword = $form_state->getValue('query') ?? '';

    // Build category name from current category parts.
    $category_name = '';
    if (isset($current_category->parts) && count($current_category->parts) > 0) {
      $category_name = implode('/', $current_category->parts);
    }

    // Build search query by combining category, keyword, and file type filters.
    $query_parts = [];
    if (!empty($category_name)) {
      $query_parts[] = "cat:($category_name)";
    }
    if (!empty($keyword)) {
      $query_parts[] = $keyword;
    }
    if (!empty($filter_type)) {
      $query_parts[] = $filter_type;
    }

    // Execute search and get results.
    $final_query = implode(' ', $query_parts);
    try {
      $search_results = $client->search($final_query, $sort_by, $num_per_page, $offset, ['thumbnails', 'embeds']);
      $items = $search_results['items'] ?? [];
      $total_asset = $search_results['total_count'] ?? 0;
    }
    catch (\Exception $e) {
      // Check if this is an authentication error.
      if (strpos($e->getMessage(), '401') !== FALSE || strpos($e->getMessage(), 'Unauthorized') !== FALSE || strpos($e->getMessage(), 'authentication') !== FALSE) {
        // User is not authenticated - show authentication message.
        $return_link = Url::fromRoute('acquia_dam.user_auth', ['uid' => $this->user->id()], ['absolute' => TRUE])->toString();
        $auth_url = $this->authService->generateAuthUrl($return_link);
        if ($auth_url) {
          $auth_link = Url::fromUri($auth_url, ['attributes' => ['target' => '_blank']]);
          $message = $this->t('Authentication required. Please %authenticate to browse Acquia DAM assets.', [
            '%authenticate' => Link::fromTextAndUrl("authenticate", $auth_link)->toString(),
          ]);
        }
        else {
          $message = $this->t('Acquia DAM authentication failed. Please contact your administrator.');
        }

        $form['message'] = [
          '#theme' => 'asset_browser_message',
          '#message' => $message,
          '#attached' => ['library' => ['entity_browser_acquia_dam/asset_browser']],
        ];
        return $form;
      }
      else {
        // Show generic error for other types of failures.
        $form['message'] = [
          '#theme' => 'asset_browser_message',
          '#message' => $this->t('Failed to load assets: @error', ['@error' => $e->getMessage()]),
          '#attached' => ['library' => ['entity_browser_acquia_dam/asset_browser']],
        ];
        return $form;
      }
    }

    // Load categories for the filter dropdown.
    try {
      $categories_response = $client->getCategories();
      $categories = $categories_response['items'] ?? [];
    }
    catch (\Exception $e) {
      // If categories fail to load, continue with empty categories.
      $categories = [];
    }

    // Build form components.
    $form += $this->getFilterSort($categories, $current_category);
    $form += $this->getBreadcrumb($current_category);

    // Create asset container.
    $form['asset-container'] = [
      '#type' => 'container',
      '#acquia_dam_category_id' => $current_category->id ?? NULL,
      '#attributes' => [
        'class' => ['acquia-dam-asset-browser'],
        'id' => 'acquia-dam-asset-container',
      ],
    ];

    // Build assets array for checkboxes.
    $assets = [];
    if (!empty($items)) {
      foreach ($items as $category_item) {
        $assets[$category_item['id']] = $this->layoutMediaEntity($category_item);
      }
    }

    // Add assets checkboxes to form.
    $form['asset-container']['assets'] = [
      '#type' => 'checkboxes',
      '#theme_wrappers' => ['checkboxes__acquia_dam_assets'],
      '#title_display' => 'invisible',
      '#options' => $assets,
      '#attached' => ['library' => ['entity_browser_acquia_dam/asset_browser']],
    ];

    // Add pagination if needed.
    if ($total_asset > $num_per_page) {
      $form['pager-container'] = $this->getPager($total_asset, $page, $num_per_page, $page_type, $current_category);
    }

    return $form;
  }

  /**
   * Builds the filter and sort form elements.
   *
   * @param array $categories
   *   Array of available categories from Acquia DAM.
   * @param object|null $current_category
   *   The currently selected category object.
   *
   * @return array
   *   Form array containing filter and sort elements.
   */
  public function getFilterSort($categories = [], $current_category = NULL) {
    // Create filter container with flexbox layout.
    $form['filter-sort-container'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['filter-sort-container']],
    ];

    // Search field for keyword filtering.
    $form['filter-sort-container']['query'] = [
      '#type' => 'textfield',
      '#title' => 'Search',
      '#size' => 24,
    ];

    // Category dropdown using widen_categories widget.
    if (!empty($categories)) {
      $form['filter-sort-container']['category'] = [
        '#type' => 'widen_categories',
        '#title' => 'Category',
        '#default_value' => 'all',
        '#ajax' => [
          'callback' => [$this, 'ajaxCategoryChange'],
          'wrapper' => 'acquia-dam-widget-form',
          'method' => 'replace',
        ],
      ];
    }

    // Get the configured media type to determine default format filter.
    $default_format = '';
    $media_type_id = $this->configuration['media_type'];
    $media_type_label = '';

    if ($media_type_id !== NULL && $media_type_id !== '') {
      $media_type = $this->entityTypeManager->getStorage('media_type')->load($this->configuration['media_type']);
      if ($media_type) {
        $media_type_label = $media_type->label();
        $default_format = $this->getDefaultFormatForMediaTypeId($media_type->id());
      }
    }

    // File format filter dropdown - simplified approach.
    $strict_restriction = $this->configuration['strict_format_restriction'];
    if ($strict_restriction && !empty($default_format)) {
      // When strict restriction is enabled, only show the relevant format
      // option.
      $format_label = ucfirst($default_format);
      $format_options = [
        $default_format => $format_label,
      ];

      $form['filter-sort-container']['format_type'] = [
        '#type' => 'select',
        '#title' => 'File format',
        '#options' => $format_options,
        '#default_value' => $default_format,
        '#description' => $this->t('Only @media_type files are allowed for this widget.', ['@media_type' => $media_type_label]),
        '#disabled' => TRUE,
        '#attributes' => [
          'class' => ['format-filter-restricted'],
        ],
      ];
    }
    else {
      // Show all format options when strict restriction is disabled.
      $format_options = $this->getAllFormatOptions();

      $form['filter-sort-container']['format_type'] = [
        '#type' => 'select',
        '#title' => 'File format',
        '#options' => $format_options,
        '#default_value' => $default_format,
      ];
    }

    // Sort by field dropdown.
    $form['filter-sort-container']['sortby'] = [
      '#type' => 'select',
      '#title' => 'Sort by',
      '#options' => [
        'filename' => $this->t('File name'),
        'created_date' => $this->t('Date created'),
        'last_update_date' => $this->t('Date modified'),
      ],
      '#default_value' => 'created_date',
    ];

    // Sort direction dropdown.
    $form['filter-sort-container']['sortdir'] = [
      '#type' => 'select',
      '#title' => 'Sort direction',
      '#options' => [
        'asc' => $this->t('Ascending'),
        'desc' => $this->t('Descending'),
      ],
      '#default_value' => 'desc',
    ];

    // Apply and Reset buttons.
    $form['filter-sort-container']['filter-sort-submit'] = [
      '#type' => 'button',
      '#value' => 'Apply',
      '#name' => 'filter_sort_submit',
    ];

    $form['filter-sort-container']['filter-sort-reset'] = [
      '#type' => 'button',
      '#value' => 'Reset',
      '#name' => 'filter_sort_reset',
    ];

    return $form;
  }

  /**
   * Returns the default ft value for the configured media type.
   */
  protected function getDefaultFormatForConfiguredMediaType() {
    $media_type_id = $this->configuration['media_type'];
    if ($media_type_id === NULL || $media_type_id === '') {
      return '';
    }
    $media_type = $this->entityTypeManager->getStorage('media_type')->load($media_type_id);
    if (!$media_type) {
      return '';
    }
    return $this->getDefaultFormatForMediaTypeId($media_type->id());
  }

  /**
   * Maps a media type machine name to an Acquia DAM ft value.
   */
  protected function getDefaultFormatForMediaTypeId($media_type_id) {
    if (strpos($media_type_id, 'image') !== FALSE || strpos($media_type_id, 'photo') !== FALSE) {
      return 'image';
    }
    if (strpos($media_type_id, 'video') !== FALSE) {
      return 'video';
    }
    if (strpos($media_type_id, 'audio') !== FALSE || strpos($media_type_id, 'sound') !== FALSE) {
      return 'audio';
    }
    if (strpos($media_type_id, 'document') !== FALSE || strpos($media_type_id, 'doc') !== FALSE || strpos($media_type_id, 'spreadsheet') !== FALSE) {
      return 'office';
    }
    if (strpos($media_type_id, 'pdf') !== FALSE) {
      return 'pdf';
    }
    if (strpos($media_type_id, 'generic') !== FALSE) {
      return 'generic_binary';
    }
    if (strpos($media_type_id, 'archive') !== FALSE || strpos($media_type_id, 'zip') !== FALSE) {
      return 'project_archive';
    }
    return '';
  }

  /**
   * Returns all format options for the filter dropdown.
   */
  protected function getAllFormatOptions() {
    return [
      '' => $this->t('All formats'),
      'image' => $this->t('Image'),
      'video' => $this->t('Video'),
      'audio' => $this->t('Audio'),
      'office' => $this->t('Documents'),
      'pdf' => $this->t('PDF'),
      'generic_binary' => $this->t('Generic'),
      'project_archive' => $this->t('Archive'),
    ];
  }

  /**
   * Builds the breadcrumb navigation for category hierarchy.
   *
   * @param object $category
   *   The current category object containing parts array.
   *
   * @return array
   *   Form array containing breadcrumb navigation elements.
   */
  public function getBreadcrumb($category) {
    // Create breadcrumb container.
    $form['breadcrumb-container'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['breadcrumb', 'acquia-dam-browser-breadcrumb-container']],
    ];

    // Add Home breadcrumb button.
    $level = [];
    $form['breadcrumb-container'][0] = [
      '#type' => 'button',
      '#value' => "Home",
      '#name' => 'breadcrumb',
      '#category_name' => NULL,
      '#parts' => $level,
      '#prefix' => '<li>',
      '#suffix' => '</li>',
      '#attributes' => ['class' => ['acquia-dam-browser-breadcrumb']],
    ];

    // Add breadcrumb buttons for each category level.
    foreach ($category->parts as $key => $category_name) {
      $level[] = $category_name;
      $key++;
      $form['breadcrumb-container'][$key] = [
        '#type' => 'button',
        '#value' => $category_name,
        '#category_name' => $category_name,
        '#name' => 'breadcrumb',
        '#parts' => $level,
        '#prefix' => '<li>',
        '#suffix' => '</li>',
        '#attributes' => ['class' => ['acquia-dam-browser-breadcrumb']],
      ];
    }

    return $form;
  }

  /**
   * Renders a single Acquia DAM asset for display in the browser.
   *
   * @param array $acquia_dam_asset
   *   The asset data from Acquia DAM API.
   *
   * @return string
   *   HTML markup for the asset display.
   */
  public function layoutMediaEntity($acquia_dam_asset) {
    $asset_id = $acquia_dam_asset['id'];

    // Check if a media entity already exists for this asset.
    $media_storage = $this->entityTypeManager->getStorage('media');
    $query = $media_storage->getQuery()
      ->condition('acquia_dam_asset_id', $asset_id)
      ->range(0, 1)
      ->accessCheck(TRUE);

    $media_ids = $query->execute();

    if (!empty($media_ids)) {
      // Use existing media entity with media_library view mode.
      $media_entity = $media_storage->load(reset($media_ids));
      $view_builder = $this->entityTypeManager->getViewBuilder('media');
      $render_array = $view_builder->view($media_entity, 'media_library');
      return $this->renderer->render($render_array);
    }
    else {
      // Create custom HTML for new assets.
      $assetName = $acquia_dam_asset['filename'];

      // Find the best available thumbnail URL.
      $thumbnail_url = '';
      if (!empty($acquia_dam_asset['thumbnails'])) {
        // Try different thumbnail sizes in order of preference.
        if (isset($acquia_dam_asset['thumbnails']['300px']['url'])) {
          $thumbnail_url = $acquia_dam_asset['thumbnails']['300px']['url'];
        }
        elseif (isset($acquia_dam_asset['thumbnails']['160px']['url'])) {
          $thumbnail_url = $acquia_dam_asset['thumbnails']['160px']['url'];
        }
        elseif (isset($acquia_dam_asset['thumbnails']['125px']['url'])) {
          $thumbnail_url = $acquia_dam_asset['thumbnails']['125px']['url'];
        }
        elseif (isset($acquia_dam_asset['thumbnails']['600px']['url'])) {
          $thumbnail_url = $acquia_dam_asset['thumbnails']['600px']['url'];
        }
      }
      elseif (!empty($acquia_dam_asset['embeds'])) {
        // Fallback to embeds if thumbnails are not available.
        if (isset($acquia_dam_asset['embeds']['Thumbnil']['url'])) {
          $thumbnail_url = $acquia_dam_asset['embeds']['Thumbnil']['url'];
        }
        elseif (isset($acquia_dam_asset['embeds']['640px-landscape']['url'])) {
          $thumbnail_url = $acquia_dam_asset['embeds']['640px-landscape']['url'];
        }
        elseif (isset($acquia_dam_asset['embeds']['640px-portrait']['url'])) {
          $thumbnail_url = $acquia_dam_asset['embeds']['640px-portrait']['url'];
        }
        elseif (isset($acquia_dam_asset['embeds']['original']['url'])) {
          $thumbnail_url = $acquia_dam_asset['embeds']['original']['url'];
        }
      }

      // Ensure thumbnail URL is absolute.
      if (!empty($thumbnail_url) && !preg_match('/^https?:\/\//', $thumbnail_url)) {
        $thumbnail_url = 'https://api.widencollective.com' . $thumbnail_url;
      }

      // Build thumbnail HTML.
      if (!empty($thumbnail_url)) {
        $thumbnail = '<div class="acquia-dam-asset-thumb"><img src="' . $thumbnail_url . '" alt="' . $assetName . '" style="max-width: 100%; height: auto;" /></div>';
      }
      else {
        $thumbnail = '<div class="acquia-dam-asset-thumb"><span class="acquia-dam-browser-empty">No preview available.</span></div>';
      }

      // Return complete asset HTML structure.
      $element = '<div class="acquia-dam-asset-checkbox">' . $thumbnail . '<div class="acquia-dam-asset-details"><p class="acquia-dam-asset-filename">' . $assetName . '</p></div></div>';
      return $element;
    }
  }

  /**
   * Builds pagination controls for asset browsing.
   *
   * @param int $total_count
   *   Total number of assets available.
   * @param int $page
   *   Current page number (0-based).
   * @param int $num_per_page
   *   Number of assets per page.
   * @param string $page_type
   *   Type of page (listing or search).
   * @param object|null $category
   *   Current category object for maintaining context.
   *
   * @return array
   *   Form array containing pagination controls.
   */
  public function getPager($total_count, $page, $num_per_page, $page_type = "listing", $category = NULL) {
    // Create pagination container.
    $pager = [
      '#type' => 'container',
      '#page' => $page,
      '#attributes' => ['class' => ['acquia-dam-browser-pager-container']],
    ];

    // Add first and previous buttons if not on first page.
    if ($page > 0) {
      $pager['first'] = [
        '#type' => 'button',
        '#value' => '<<',
        '#name' => 'acquia_dam_pager',
        '#page_type' => $page_type,
        '#current_category' => $category,
        '#acquia_dam_page' => 0,
        '#attributes' => ['class' => ['page-button', 'page-first']],
      ];
      $pager['previous'] = [
        '#type' => 'button',
        '#value' => '<',
        '#name' => 'acquia_dam_pager',
        '#page_type' => $page_type,
        '#acquia_dam_page' => $page - 1,
        '#current_category' => $category,
        '#attributes' => ['class' => ['page-button', 'page-previous']],
      ];
    }

    // Calculate pagination range.
    $last_page = floor(($total_count - 1) / $num_per_page);
    $start_page = max(0, $page - 4);
    $end_page = min($start_page + 9, $last_page);

    // Add numbered page buttons.
    for ($i = $start_page; $i <= $end_page; $i++) {
      $pager['page_' . $i] = [
        '#type' => 'button',
        '#value' => $i + 1,
        '#name' => 'acquia_dam_pager',
        '#page_type' => $page_type,
        '#acquia_dam_page' => $i,
        '#current_category' => $category,
        '#attributes' => ['class' => [($i == $page ? 'page-current' : ''), 'page-button']],
      ];
    }

    // Add next and last buttons if not on last page.
    if ($end_page > $page) {
      $pager['next'] = [
        '#type' => 'button',
        '#value' => '>',
        '#name' => 'acquia_dam_pager',
        '#current_category' => $category,
        '#page_type' => $page_type,
        '#acquia_dam_page' => $page + 1,
        '#attributes' => ['class' => ['page-button', 'page-next']],
      ];
      $pager['last'] = [
        '#type' => 'button',
        '#value' => '>>',
        '#name' => 'acquia_dam_pager',
        '#current_category' => $category,
        '#acquia_dam_page' => $last_page,
        '#page_type' => $page_type,
        '#attributes' => ['class' => ['page-button', 'page-last']],
      ];
    }
    return $pager;
  }

  /**
   * {@inheritdoc}
   */
  public function validate(array &$form, FormStateInterface $form_state) {
    // Only validate when the main submit button is clicked.
    if (!empty($form_state->getTriggeringElement()['#eb_widget_main_submit'])) {
      // Load the configured media type.
      $media_bundle = $this->entityTypeManager->getStorage('media_type')
        ->load($this->configuration['media_type']);

      // Get the source field that stores the asset ID.
      $source_field_definition = $media_bundle->getSource()->getSourceFieldDefinition($media_bundle);
      $source_field_name = $source_field_definition->getName();

      // Validate that the source field exists in the media type.
      $field_definitions = $this->entityFieldManager->getFieldDefinitions('media', $media_bundle->id());
      if (!isset($field_definitions[$source_field_name])) {
        $message = $this->t(
          'Missing Acquia DAM asset field mapping. Source field "@field" not found. Available fields: @fields. Check your media configuration.',
          [
            '@field' => $source_field_name,
            '@fields' => implode(', ', array_keys($field_definitions)),
          ]
        );
        $form_state->setError($form['widget']['asset-container']['assets'], $message);
        return;
      }

      // Get selected assets and validate cardinality.
      $assets = array_filter($form_state->getValue('assets'));
      $field_cardinality = $form_state->get([
        'entity_browser',
        'validators',
        'cardinality',
        'cardinality',
      ]);

      // Ensure at least one asset is selected.
      if (!count($assets)) {
        $form_state->setError($form['widget']['asset-container'], $this->t('Please select an asset.'));
      }

      // Check cardinality limits.
      if ($field_cardinality > 0 && count($assets) > $field_cardinality) {
        $message = $this->formatPlural($field_cardinality, 'You can not select more than 1 entity.', 'You can not select more than @count entities.');
        $form_state->setError($form['widget']['asset-container']['assets'], $message);
      }

      // Validate that selected assets exist in Acquia DAM.
      $client = $this->clientFactory->getUserClient();
      $dam_assets = [];
      foreach ($assets as $asset_id) {
        try {
          $dam_assets[] = $client->getAsset($asset_id);
        }
        catch (\Exception $e) {
          // Skip invalid assets.
          continue;
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submit(array &$element, array &$form, FormStateInterface $form_state) {
    $assets = [];
    if (!empty($form_state->getTriggeringElement()['#eb_widget_main_submit'])) {
      $assets = $this->prepareEntities($form, $form_state);
    }
    $this->selectEntities($assets, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  protected function prepareEntities(array $form, FormStateInterface $form_state) {
    // Get selected asset IDs from form state.
    $current_selections = $form_state->getValue('current_selections', []);
    $assets = $form_state->getValue('assets', []);
    $filtered_assets = array_filter($assets);
    $asset_ids = is_array($current_selections) ? $current_selections : [];
    $asset_ids = array_merge($asset_ids, is_array($filtered_assets) ? $filtered_assets : []);

    $media_type = $this->entityTypeManager->getStorage('media_type')->load($this->configuration['media_type']);
    $source_field = $media_type->getSource()->getSourceFieldDefinition($media_type)->getName();

    // Check for existing media entities to avoid duplicates.
    // Query for existing assets using the asset_id property within the field.
    $existing_ids = $this->entityTypeManager->getStorage('media')
      ->getQuery()
      ->condition('bundle', $media_type->id())
      ->condition($source_field . '.asset_id', $asset_ids, 'IN')
      ->accessCheck(TRUE)
      ->execute();
    $entities = $this->entityTypeManager->getStorage('media')->loadMultiple($existing_ids);

    // Create a map of existing asset IDs to their media entities.
    $existing_asset_map = [];
    foreach ($entities as $entity) {
      $field_item = $entity->get($source_field)->first();
      if ($field_item && !$field_item->isEmpty()) {
        $asset_id = $field_item->get('asset_id')->getValue();
        if ($asset_id) {
          $existing_asset_map[$asset_id] = $entity;
        }
      }
    }

    // Filter out existing assets from the list to avoid duplicates.
    $new_asset_ids = [];
    foreach ($asset_ids as $asset_id) {
      if (!isset($existing_asset_map[$asset_id])) {
        $new_asset_ids[] = $asset_id;
      }
    }

    // Fetch asset data from Acquia DAM for new assets only.
    $client = $this->clientFactory->getUserClient();
    $assets = [];
    foreach ($new_asset_ids as $asset_id) {
      try {
        $assets[] = $client->getAsset($asset_id);
      }
      catch (\Exception $e) {
        // Check if this is an authentication error.
        if (strpos($e->getMessage(), '401') !== FALSE || strpos($e->getMessage(), 'Unauthorized') !== FALSE || strpos($e->getMessage(), 'authentication') !== FALSE) {
          // User is not authenticated - show authentication message.
          $return_link = Url::fromRoute('acquia_dam.user_auth', ['uid' => $this->user->id()], ['absolute' => TRUE])->toString();
          $auth_url = $this->authService->generateAuthUrl($return_link);
          if ($auth_url) {
            $auth_link = Url::fromUri($auth_url, ['attributes' => ['target' => '_blank']]);
            $this->messenger()->addError($this->t('Authentication required. Please %authenticate to access Acquia DAM assets.', [
              '%authenticate' => Link::fromTextAndUrl("authenticate", $auth_link)->toString(),
            ]));
          }
          else {
            $this->messenger()->addError($this->t('Acquia DAM authentication failed. Please contact your administrator.'));
          }
          return [];
        }
        else {
          // Skip invalid assets for other types of errors.
          $this->messenger()->addWarning($this->t('Skipped asset @asset_id due to error: @error', [
            '@asset_id' => $asset_id,
            '@error' => $e->getMessage(),
          ]));
          continue;
        }
      }
    }

    // Create new media entities for assets that don't exist yet.
    foreach ($assets as $asset) {
      $entity_values = [
        'bundle' => $media_type->id(),
        'uid' => $this->user->id(),
        'langcode' => $this->languageManager->getCurrentLanguage()->getId(),
        'status' => 1,
        'name' => $asset['filename'],
        $source_field => [
          'asset_id' => $asset['id'],
        ],
        'created' => strtotime($asset['created_date']),
        'changed' => strtotime($asset['last_update_date']),
      ];

      $entity = $this->entityTypeManager->getStorage('media')->create($entity_values);
      $entity->save();
      $entity = $this->entityTypeManager->getStorage('media')->load($entity->id());
      $entities[] = $entity;
    }

    return $entities;
  }

  /**
   * AJAX callback for category selection changes.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return array
   *   The form array to be rebuilt.
   */
  public function ajaxCategoryChange(array &$form, FormStateInterface $form_state) {
    $selected_category = $form_state->getValue('category');

    // Store the selected category path in form state.
    if ($selected_category && $selected_category !== 'all') {
      $form_state->set('selected_category_path', $selected_category);
    }
    else {
      $form_state->set('selected_category_path', NULL);
    }

    // Trigger form rebuild to update the asset list.
    $form_state->setRebuild();
    return $form;
  }

}
