<?php

namespace Drupal\entity_browser_acquia_dam\Plugin\EntityBrowser\Widget;

use Drupal\acquia_dam\AcquiadamAuthService;
use Drupal\acquia_dam\AssetRepository;
use Drupal\acquia_dam\Client\AcquiaDamClientFactory;
use Drupal\acquia_dam\MediaTypeResolver;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\entity_browser\WidgetBase;
use Drupal\entity_browser\WidgetValidationManager;
use Drupal\media\MediaSourceManager;
use Drupal\user\UserDataInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Entity browser widget for uploading assets to Acquia DAM.
 *
 * Provides a widget for uploading files to Acquia DAM and creating media
 * entities within entity browsers. Supports upload profile selection and file
 * validation.
 *
 * @EntityBrowserWidget(
 *   id = "entity_browser_acquia_dam_upload",
 *   label = @Translation("Acquia DAM Upload"),
 *   description = @Translation("Upload assets to Acquia DAM and create media entities"),
 *   auto_select = FALSE
 * )
 */
class AcquiaDamUpload extends WidgetBase {

  /**
   * The DAM client factory.
   *
   * @var \Drupal\acquia_dam\Client\AcquiaDamClientFactory
   */
  protected $clientFactory;

  /**
   * The current user account.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $user;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The media source manager.
   *
   * @var \Drupal\media\MediaSourceManager
   */
  protected $sourceManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The user data manager.
   *
   * @var \Drupal\user\UserDataInterface
   */
  protected $userData;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The media type resolver.
   *
   * @var \Drupal\acquia_dam\MediaTypeResolver
   */
  protected $mediaTypeResolver;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The Acquia DAM auth service.
   *
   * @var \Drupal\acquia_dam\AcquiadamAuthService
   */
  protected $authService;

  /**
   * The asset repository.
   *
   * @var \Drupal\acquia_dam\AssetRepository
   */
  protected $assetRepository;

  /**
   * Constructs a new AcquiaDamUpload widget.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event dispatcher.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\entity_browser\WidgetValidationManager $validation_manager
   *   The widget validation manager.
   * @param \Drupal\acquia_dam\Client\AcquiaDamClientFactory $client_factory
   *   The DAM client factory.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The current user account.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\media\MediaSourceManager $sourceManager
   *   The media source manager.
   * @param \Drupal\user\UserDataInterface $userData
   *   The user data manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config
   *   The config factory.
   * @param \Drupal\acquia_dam\MediaTypeResolver $media_type_resolver
   *   The media type resolver.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\acquia_dam\AcquiadamAuthService $auth_service
   *   The Acquia DAM auth service.
   * @param \Drupal\acquia_dam\AssetRepository $asset_repository
   *   The asset repository.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EventDispatcherInterface $event_dispatcher, EntityTypeManagerInterface $entity_type_manager, EntityFieldManagerInterface $entity_field_manager, WidgetValidationManager $validation_manager, AcquiaDamClientFactory $client_factory, AccountInterface $account, LanguageManagerInterface $languageManager, ModuleHandlerInterface $moduleHandler, MediaSourceManager $sourceManager, UserDataInterface $userData, RequestStack $requestStack, ConfigFactoryInterface $config, MediaTypeResolver $media_type_resolver, RendererInterface $renderer, AcquiadamAuthService $auth_service, AssetRepository $asset_repository) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $event_dispatcher, $entity_type_manager, $validation_manager);
    $this->clientFactory = $client_factory;
    $this->user = $account;
    $this->languageManager = $languageManager;
    $this->moduleHandler = $moduleHandler;
    $this->sourceManager = $sourceManager;
    $this->entityFieldManager = $entity_field_manager;
    $this->userData = $userData;
    $this->requestStack = $requestStack;
    $this->config = $config;
    $this->mediaTypeResolver = $media_type_resolver;
    $this->renderer = $renderer;
    $this->authService = $auth_service;
    $this->assetRepository = $asset_repository;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('event_dispatcher'),
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('plugin.manager.entity_browser.widget_validation'),
      $container->get('acquia_dam.client.factory'),
      $container->get('current_user'),
      $container->get('language_manager'),
      $container->get('module_handler'),
      $container->get('plugin.manager.media.source'),
      $container->get('user.data'),
      $container->get('request_stack'),
      $container->get('config.factory'),
      $container->get('acquia_dam.media_type_resolver'),
      $container->get('renderer'),
      $container->get('acquia_dam.authentication_service'),
      $container->get('acquia_dam.asset_repository')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    // Submit text configuration.
    $form['submit_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Submit button text'),
      '#default_value' => $this->configuration['submit_text'],
      '#description' => $this->t('Text to display on the upload submit button.'),
    ];

    // Allow profile selection configuration.
    $form['allow_profile_selection'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow upload profile selection'),
      '#default_value' => $this->configuration['allow_profile_selection'],
      '#description' => $this->t('If checked, users can select which upload profile to use. If unchecked, the default profile from configuration will be used.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'submit_text' => $this->t('Upload to Acquia DAM'),
      'allow_profile_selection' => TRUE,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function getForm(array &$original_form, FormStateInterface $form_state, array $additional_widget_parameters) {
    // Check if the user has access to Acquia DAM.
    $client = $this->clientFactory->getUserClient();
    try {
      $client->getCategories();
    }
    catch (\Exception $e) {
      // User is not authenticated or DAM is not configured.
      $return_link = Url::fromRoute('acquia_dam.user_auth', ['uid' => $this->user->id()], ['absolute' => TRUE])->toString();
      $auth_url = $this->authService->generateAuthUrl($return_link);
      if ($auth_url) {
        $auth_link = Url::fromUri($auth_url, ['attributes' => ['target' => '_blank']]);
        $message = $this->t('You are not authenticated. Please %authenticate to upload assets to Acquia DAM.', [
          '%authenticate' => Link::fromTextAndUrl("Authenticate", $auth_link)->toString(),
        ]);
      }
      else {
        $message = $this->t('Acquia DAM module is not configured yet. Please contact your administrator to do so.');
        if ($this->user->hasPermission('administer site configuration')) {
          $message = $this->t('Acquia DAM module is not configured yet. Please %config it to start uploading assets.', [
            '%config' => Link::createFromRoute($this->t('configure'), 'acquia_dam.config', [], ['attributes' => ['target' => '_blank']])->toString(),
          ]);
        }
      }

      $form['message'] = [
        '#theme' => 'asset_browser_message',
        '#message' => $message,
        '#attached' => ['library' => ['entity_browser_acquia_dam/asset_browser']],
      ];
      return $form;
    }

    // Build the main form.
    $form = parent::getForm($original_form, $form_state, $additional_widget_parameters);
    $form['#attributes']['id'] = 'acquia-dam-upload-widget-form';
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';

    // File upload field - no extension restrictions since Acquia DAM handles
    // media type detection.
    $form['asset'] = [
      '#title' => $this->t('Upload asset'),
      '#type' => 'file',
      '#description' => $this->t('Select the media asset to upload to Acquia DAM. The system will automatically determine the appropriate media type based on the file properties.'),
      '#required' => TRUE,
      '#upload_validators' => [
        'file_validate_extensions' => ['jpg jpeg png gif bmp tiff webp svg mp4 mov avi wmv flv webm mkv 3gp mp3 wav aac ogg flac m4a wma pdf doc docx xls xlsx ppt pptx txt rtf odt ods odp zip rar 7z tar gz bz2'],
      ],
    ];

    // Optional filename override.
    $form['filename'] = [
      '#title' => $this->t('Filename (optional)'),
      '#description' => $this->t('Optionally give the asset its own title. Defaults to the uploaded filename.'),
      '#type' => 'textfield',
    ];

    // Upload profile selection.
    if ($this->configuration['allow_profile_selection']) {
      try {
        $client = $this->clientFactory->getSiteClient();
        $profiles = $client->getUploadProfiles();
        $profile_options = [];
        foreach ($profiles as $profile) {
          $profile_options[$profile['name']] = $profile['name'];
        }

        // Get default profile from config.
        $config = $this->config->get('acquia_dam.settings');
        $default_profile = $config->get('upload_profile');

        $form['upload_profile'] = [
          '#type' => 'select',
          '#title' => $this->t('Upload profile'),
          '#options' => $profile_options,
          '#default_value' => $default_profile,
          '#description' => $this->t("Profiles control a new asset's initial metadata and security defaults when uploaded."),
          '#required' => TRUE,
        ];
      }
      catch (\Exception $e) {
        // Check if this is an authentication error.
        if (strpos($e->getMessage(), '401') !== FALSE || strpos($e->getMessage(), 'Unauthorized') !== FALSE || strpos($e->getMessage(), 'authentication') !== FALSE) {
          // User is not authenticated - show authentication message.
          $return_link = Url::fromRoute('acquia_dam.user_auth', ['uid' => $this->user->id()], ['absolute' => TRUE])->toString();
          $auth_url = $this->authService->generateAuthUrl($return_link);
          if ($auth_url) {
            $auth_link = Url::fromUri($auth_url, ['attributes' => ['target' => '_blank']]);
            $message = $this->t('Authentication required. Please %authenticate to upload assets to Acquia DAM.', [
              '%authenticate' => Link::fromTextAndUrl("authenticate", $auth_link)->toString(),
            ]);
          }
          else {
            $message = $this->t('Acquia DAM authentication failed. Please contact your administrator.');
          }

          $form['message'] = [
            '#theme' => 'asset_browser_message',
            '#message' => $message,
            '#attached' => ['library' => ['entity_browser_acquia_dam/asset_browser']],
          ];
          return $form;
        }
        else {
          // If we can't get profiles for other reasons, hide the field and use
          // default.
          $form['upload_profile'] = [
            '#type' => 'hidden',
            '#value' => $this->config->get('acquia_dam.settings')->get('upload_profile'),
          ];
        }
      }
    }
    else {
      // Use default profile from config.
      $form['upload_profile'] = [
        '#type' => 'hidden',
        '#value' => $this->config->get('acquia_dam.settings')->get('upload_profile'),
      ];
    }

    return $form;
  }

  /**
   * AJAX callback for upload submission.
   */
  public function ajaxUpload(array &$form, FormStateInterface $form_state) {
    if ($form_state->hasAnyErrors()) {
      return $form;
    }

    // Process the upload directly.
    $entities = $this->prepareEntities($form, $form_state);
    if (!empty($entities)) {
      $this->selectEntities($entities, $form_state);
    }

    // Return the form for potential additional uploads.
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validate(array &$form, FormStateInterface $form_state) {
    // Check if a file was uploaded.
    $request = $this->requestStack->getCurrentRequest();
    $file = $request->files->get('files')['asset'] ?? NULL;

    if (!$file) {
      $form_state->setErrorByName('asset', $this->t('No file uploaded.'));
      return;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submit(array &$element, array &$form, FormStateInterface $form_state) {
    $entities = $this->prepareEntities($form, $form_state);
    $this->selectEntities($entities, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  protected function prepareEntities(array $form, FormStateInterface $form_state) {
    $request = $this->requestStack->getCurrentRequest();

    /** @var \Symfony\Component\HttpFoundation\File\UploadedFile $file */
    $file = $request->files->get('files')['asset'];
    $file_content = file_get_contents($file->getPathname());

    $filename = $form_state->getValue('filename');
    if (empty($filename)) {
      $filename = $file->getClientOriginalName();
    }

    $upload_profile = $form_state->getValue('upload_profile');

    try {
      $client = $this->clientFactory->getSiteClient();
      $response = $client->uploadAsset($filename, $file_content, $upload_profile);

      $asset_api_url_bits = explode('/', $response['_links']['self']);
      $asset_id = array_pop($asset_api_url_bits);

      // Wait for asset processing.
      sleep(2);

      // Import the asset as a media entity - Acquia DAM will automatically
      // determine the media type.
      $media_ids = $this->assetRepository->import([$asset_id]);

      if (!empty($media_ids)) {
        $media_entities = $this->entityTypeManager->getStorage('media')->loadMultiple($media_ids);

        $this->messenger()->addStatus($this->t('@filename was successfully uploaded to Acquia DAM and imported into Drupal', [
          '@filename' => $filename,
        ]));

        return $media_entities;
      }
    }
    catch (\Exception $e) {
      // Check if this is an authentication error.
      if (strpos($e->getMessage(), '401') !== FALSE || strpos($e->getMessage(), 'Unauthorized') !== FALSE || strpos($e->getMessage(), 'authentication') !== FALSE) {
        // User is not authenticated - show authentication message.
        $return_link = Url::fromRoute('acquia_dam.user_auth', ['uid' => $this->user->id()], ['absolute' => TRUE])->toString();
        $auth_url = $this->authService->generateAuthUrl($return_link);
        if ($auth_url) {
          $auth_link = Url::fromUri($auth_url, ['attributes' => ['target' => '_blank']]);
          $this->messenger()->addError($this->t('Authentication required to upload @filename. Please %authenticate to upload assets to Acquia DAM.', [
            '@filename' => $filename,
            '%authenticate' => Link::fromTextAndUrl("authenticate", $auth_link)->toString(),
          ]));
        }
        else {
          $this->messenger()->addError($this->t('Acquia DAM authentication failed. Please contact your administrator.'));
        }
      }
      else {
        // Show generic error for other types of failures.
        $this->messenger()->addError($this->t('Failed to upload @filename: @error', [
          '@filename' => $filename,
          '@error' => $e->getMessage(),
        ]));
      }
    }

    return [];
  }

}
