/**
 * @file entity_browser.table.js
 *
 * Allows the table rows to be sortable.
 */

(function ($, Drupal, Sortable) {
  /**
   * Registers behaviors related to entity reference field widget.
   */
  Drupal.behaviors.entityBrowserEntityReferenceTable = {
    attach(context) {
      document
        .querySelector('.table--widget-entity_reference_browser_table_widget')
        .querySelector('tbody').classList += ' sortable ui-sortable';
      $(context)
        .find('.table--widget-entity_reference_browser_table_widget')
        .find('tbody')
        .addClass('sortable ui-sortable');

      const sortableSelector = context.querySelectorAll(
        '.table--widget-entity_reference_browser_table_widget tbody.sortable',
      );
      let dataRow;
      let delta;
      sortableSelector.forEach(function (widget) {
        $(widget)
          .find('.item-container .handle')
          .parent()
          .addClass('handle-cell');
        Sortable.create(widget, {
          handle: '.handle',
          draggable: '.item-container',
          onEnd: function onEnd(e) {
            // Set our unsaved drag deltas in storage.
            const items = e.item.parentElement.children;
            Array.from(items).forEach((val, index) => {
              dataRow = val.getAttribute('data-entity-id');
              localStorage.setItem(dataRow, index);
            });

            return Drupal.entityBrowserEntityReferenceTable.entitiesReordered(
              widget,
            );
          },
        });
      });
      // The AJAX callback will give us a flag when we need to re-open the
      // browser, most likely due to a "Replace" button being clicked.
      if (
        typeof drupalSettings.entity_browser_reopen_browser !== 'undefined' &&
        drupalSettings.entity_browser_reopen_browser
      ) {
        const dataDrupalSelector = `[data-drupal-selector^="edit-${drupalSettings.entity_browser_reopen_browser.replace(
          /_/g,
          '-',
        )}-entity-browser-entity-browser"][data-uuid]`;
        const $launchBrowserElement = $(context).find(dataDrupalSelector);
        if (
          $launchBrowserElement.attr('data-uuid') in
            drupalSettings.entity_browser &&
          !drupalSettings.entity_browser[
            $launchBrowserElement.attr('data-uuid')
          ].auto_open
        ) {
          $launchBrowserElement.click();
        }
        // In case this is inside a fieldset closed by default, open it so the
        // user doesn't need to guess the browser is open but hidden there.
        const $fieldsetSummary = $launchBrowserElement
          .closest('details')
          .find('summary');
        if (
          $fieldsetSummary.length &&
          $fieldsetSummary.attr('aria-expanded') === 'false'
        ) {
          $fieldsetSummary.click();
        }
      }
      // After ajax ops, reorder per deltas set in storage.
      // eslint-disable-next-line
      $(document).ajaxComplete(() => {
        document.querySelectorAll('.item-container').forEach((item) => {
          dataRow = $(item).attr('data-entity-id');
          delta = localStorage.getItem(dataRow);
          $(item).attr('data-row-id', delta);
        });
        sortableSelector.forEach((widget) => {
          $(widget)
            .find('.item-container')
            .sort((a, b) => {
              return $(a).attr('data-row-id') - $(b).attr('data-row-id');
            })
            .appendTo($(widget));
        });
      });
    },
  };

  Drupal.entityBrowserEntityReferenceTable = {};

  /**
   * Reacts on sorting of entities for the table widget.
   *
   * @param {object} widget
   *   Object with the sortable area.
   */
  Drupal.entityBrowserEntityReferenceTable.entitiesReordered = function (
    widget,
  ) {
    const items = $(widget).find('.item-container');
    const ids = [];
    for (let i = 0; i < items.length; i++) {
      ids[i] = $(items[i]).attr('data-entity-id');
    }
    let currenItems = $(widget).parent().parent();
    if (
      $(widget).parents('.field--widget-entity_reference_browser_table_widget')
        .length > 0
    ) {
      currenItems = $(widget).parents(
        '.field--widget-entity_reference_browser_table_widget',
      );
    }
    currenItems
      .find('input[type*=hidden][name*="[target_id]"]')
      .val(ids.join(' '));
  };
})(jQuery, Drupal, Sortable);
