<?php

namespace Drupal\entity_browser_table\Plugin\Field\FieldWidget;

use Drupal\Component\Utility\Html;
use Drupal\content_moderation\ModerationInformationInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\entity_browser\Plugin\Field\FieldWidget\EntityReferenceBrowserWidget;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'entity_reference_browser_table_widget' widget.
 */
#[FieldWidget(
  id: 'entity_reference_browser_table_widget',
  label: new TranslatableMarkup('Entity Browser - Table'),
  field_types: ['entity_reference'],
  multiple_values: TRUE,
)]
class EntityReferenceBrowserTableWidget extends EntityReferenceBrowserWidget {

  /**
   * Language code ID.
   */
  protected string $currentLanguage;

  /**
   * The entity type bundle info service.
   */
  protected EntityTypeBundleInfoInterface $entityBundleInfo;

  /**
   * The moderation information service.
   */
  protected ?ModerationInformationInterface $moderationInfo;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->currentLanguage = $container->get('language_manager')->getCurrentLanguage()->getId();
    $instance->entityBundleInfo = $container->get('entity_type.bundle.info');
    $instance->moderationInfo = $container->has('content_moderation.moderation_information') ? $container->get('content_moderation.moderation_information') : NULL;
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return [
      'additional_fields' => ['options' => NULL],
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $form = parent::settingsForm($form, $form_state);

    $form['additional_fields'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Additional Fields'),
    ];
    $form['additional_fields']['options'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Additional Fields'),
      '#options' => [
        'status' => $this->t('Status or, if enabled, moderation status.'),
      ],
      '#weight' => '1',
      '#default_value' => $this->getSetting('additional_fields')['options'] ?? [],
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);
    $element['#attributes']['class'] = [
      'field--widget-entity_reference_browser_table_widget',
    ];
    $element['#attached']['library'][] = 'entity_browser_table/entity_browser_table';

    return $element;
  }

  /**
   * Builds the render array for displaying the current results as a table.
   *
   * @param string $details_id
   *   The ID for the details' element.
   * @param string[] $field_parents
   *   Field parents.
   * @param \Drupal\Core\Entity\ContentEntityInterface[] $entities
   *   Array of referenced entities.
   *
   * @return array
   *   The render array for the current selection.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  protected function displayCurrentSelection($details_id, array $field_parents, array $entities): array {
    $array_merged = [];
    if (!empty($entities)) {
      $table = [
        '#type' => 'table',
        '#header' => $this->buildTableHeaders(),
        '#attributes' => ['class' => ['table--widget-' . $this->getPluginId()]],
        '#empty' => $this->t('Use the buttons above to add content to this area.'),
      ];
      $array_merged = array_merge($table, $this->buildTableRows($details_id, $field_parents, $entities));

      $this->moduleHandler->alter('entity_browser_table', $array_merged, $entities, $details_id);
    }

    return $array_merged;
  }

  /**
   * Helper function to build table headers.
   *
   * @return array
   *   The header.
   */
  public function buildTableHeaders(): array {
    return array_filter(
      [
        '',
        $this->getFirstColumnHeader(),
        $this->getAdditionalFieldsColumnHeader(),
        $this->getActionColumnHeader(),
      ],
      function ($header) {
        return $header !== NULL;
      }
    );
  }

  /**
   * Helper function to build table rows.
   *
   * @param string $details_id
   *   The ID for the details' element.
   * @param string[] $field_parents
   *   Field parents.
   * @param \Drupal\Core\Entity\ContentEntityInterface[] $entities
   *   Array of referenced entities.
   *
   * @return array
   *   The table rows.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function buildTableRows(string $details_id, array $field_parents, array $entities): array {
    $field_widget_display = $this->fieldDisplayManager->createInstance(
      $this->getSetting('field_widget_display'),
      $this->getSetting('field_widget_display_settings') + [
        'entity_type' => $this->fieldDefinition->getFieldStorageDefinition()
          ->getSetting('target_type'),
      ]
    );

    $entities = array_filter($entities, function ($entity) {
      return $entity instanceof EntityInterface;
    });
    $rowData = [];
    foreach ($entities as $row_id => $entity) {
      if ($entity->hasTranslation($this->currentLanguage)) {
        $entity = $entity->getTranslation($this->currentLanguage);
      }

      $rowData[] = array_filter([
        'handle' => $this->buildSortableHandle(),
        'title-preview' => $this->getFirstColumn($entity),
        'status' => $this->getAdditionalFieldsColumn($entity),
        'actions' => [
          'edit_button' => $this->buildEditButton($entity, $details_id, $row_id, $field_parents),
          'replace_button' => $this->buildReplaceButton($entity, $entities, $details_id, $row_id, $field_parents),
          'remove_button' => $this->buildRemoveButton($entity, $details_id, $row_id, $field_parents),
        ],
        '#attributes' => [
          'class' => [
            'item-container',
            Html::getClass($field_widget_display->getPluginId()),
          ],
          'data-entity-id' => $entity->getEntityTypeId() . ':' . $entity->id(),
          'data-row-id' => $row_id,
        ],

      ]);
    }

    return $rowData;
  }

  /**
   * Build sortable handler.
   *
   * @return string[]
   *   Markup array.
   */
  public function buildSortableHandle(): array {
    return [
      '#type' => 'markup',
      '#markup' => '<span class="handle">',
    ];
  }

  /**
   * Function to build the edit button.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity of the row.
   * @param string $details_id
   *   The ID for the details' element.
   * @param int $row_id
   *   Row ID.
   * @param string[] $field_parents
   *   Field parents.
   *
   * @return array
   *   The edit button
   */
  public function buildEditButton(EntityInterface $entity, string $details_id, int $row_id, array $field_parents): array {
    return [
      '#type' => 'submit',
      '#value' => $this->t('Edit'),
      '#name' => $this->fieldDefinition->getName() . '_edit_button_' . $entity->id() . '_' . $row_id . '_' . md5(json_encode($field_parents)),
      '#ajax' => [
        'url' => Url::fromRoute(
          'entity_browser.edit_form', [
            'entity_type' => $entity->getEntityTypeId(),
            'entity' => $entity->id(),
          ]
        ),
        'options' => [
          'query' => [
            'details_id' => $details_id,
          ],
        ],
      ],
      '#attributes' => [
        'class' => ['edit-button'],
      ],
      '#access' => $this->getEditButtonAccess($entity),
    ];
  }

  /**
   * Function to build remove button.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity of the row.
   * @param string $details_id
   *   The ID for the details' element.
   * @param int $row_id
   *   Row ID.
   * @param string[] $field_parents
   *   Field parents.
   *
   * @return array
   *   The edit button.
   */
  public function buildRemoveButton(EntityInterface $entity, string $details_id, int $row_id, array $field_parents): array {
    return [
      '#type' => 'submit',
      '#value' => $this->t('Remove'),
      '#ajax' => [
        'callback' => [get_class($this), 'updateWidgetCallback'],
        'wrapper' => $details_id,
      ],
      '#submit' => [[get_class($this), 'removeItemSubmit']],
      '#name' => $this->fieldDefinition->getName() . '_remove_' . $entity->id() . '_' . $row_id . '_' . md5(json_encode($field_parents)),
      '#limit_validation_errors' => [array_merge($field_parents, [$this->fieldDefinition->getName()])],
      '#attributes' => [
        'data-entity-id' => $entity->getEntityTypeId() . ':' . $entity->id(),
        'data-row-id' => $row_id,
        'class' => ['remove-button'],
      ],
      '#access' => (bool) $this->getSetting('field_widget_remove'),
    ];
  }

  /**
   * Function to build replace button.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity of the row.
   * @param \Drupal\Core\Entity\ContentEntityInterface[] $entities
   *   Array of referenced entities.
   * @param string $details_id
   *   The ID for the details' element.
   * @param int $row_id
   *   Row ID.
   * @param string[] $field_parents
   *   Field parents.
   *
   * @return array
   *   The edit button.
   */
  public function buildReplaceButton(EntityInterface $entity, array $entities, string $details_id, int $row_id, array $field_parents): array {
    return [
      '#type' => 'submit',
      '#value' => $this->t('Replace'),
      '#ajax' => [
        'callback' => [get_class($this), 'updateWidgetCallback'],
        'wrapper' => $details_id,
      ],
      '#submit' => [[get_class($this), 'removeItemSubmit']],
      '#name' => $this->fieldDefinition->getName() . '_replace_' . $entity->id() . '_' . $row_id . '_' . md5(json_encode($field_parents)),
      '#limit_validation_errors' => [array_merge($field_parents, [$this->fieldDefinition->getName()])],
      '#attributes' => [
        'data-entity-id' => $entity->getEntityTypeId() . ':' . $entity->id(),
        'data-row-id' => $row_id,
        'class' => ['replace-button'],
      ],
      '#access' => $this->getReplaceButtonAccess($entities),
    ];
  }

  /**
   * Check edit button access.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity of the row.
   *
   * @return bool
   *   If the user has access to the edit button.
   */
  public function getEditButtonAccess(EntityInterface $entity): bool {
    $edit_button_access = $this->getSetting('field_widget_edit') && $entity->access('update', $this->currentUser);
    if ($entity->getEntityTypeId() == 'file') {
      // On file entities, the "edit" button shouldn't be visible unless
      // the module "file_entity" is present, which will allow them to be
      // edited on their own form.
      $edit_button_access &= $this->moduleHandler->moduleExists('file_entity');
    }
    return (bool) $edit_button_access;
  }

  /**
   * Check replace button access.
   *
   * @param array $entities
   *   The number of entities to in the table.
   *
   * @return bool
   *   If the user has access to the replace button.
   */
  public function getReplaceButtonAccess(array $entities): bool {
    return $this->getSetting('field_widget_replace') && (count($entities) === 1);
  }

  /**
   * Get the first column.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity of the row.
   *
   * @return array
   *   First column markup array.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function getFirstColumn(EntityInterface $entity): array {
    $value = $this->getFieldWidgetDisplay()->view($entity);
    if (is_string($value)) {
      if ($value == ' ') {
        $value = $this->t('<i>No title set</i>');
      }
      $value = ['#markup' => $value];
    }
    return $value;
  }

  /**
   * Get additional columns.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity of the row.
   *
   * @return array
   *   Additional column array, empty by default.
   */
  public function getAdditionalFieldsColumn(EntityInterface $entity): array {
    if (isset($this->getAdditionalFields()['status']) === FALSE) {
      return [];
    }

    $status = $this->moderationInfo && $this->moderationInfo->isModeratedEntity($entity)
      ? $entity->get('moderation_state')->value
      : ($entity->get('status')->value === '0' ? 'Unpublished' : 'Published');

    return [
      '#markup' => '<span class="moderation-status">' . $this->t(':status', [':status' => $status]) . '</span>',
    ];
  }

  /**
   * Get the first column header label.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The label.
   */
  public function getFirstColumnHeader(): TranslatableMarkup {
    return $this->getSetting('field_widget_display') == 'rendered_entity'
      ? $this->t('Thumbnail')
      : $this->t('Title');
  }

  /**
   * Get the additional column header label.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup|null
   *   The label. NULL if empty.
   */
  public function getAdditionalFieldsColumnHeader(): ?TranslatableMarkup {
    return isset($this->getAdditionalFields()['status']) ? $this->t('Status') : NULL;
  }

  /**
   * Get the action column header.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup|null
   *   The label. NULL if empty.
   */
  public function getActionColumnHeader(): ?TranslatableMarkup {
    return $this->getSetting('field_widget_edit') || $this->getSetting('field_widget_remove') || $this->getSetting('field_widget_replace')
      ? $this->t('Action')
      : NULL;
  }

  /**
   * Get the field widget display.
   *
   * @return object
   *   The field widget display.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function getFieldWidgetDisplay(): object {
    return $this->fieldDisplayManager->createInstance(
      $this->getSetting('field_widget_display'), $this->getSetting('field_widget_display_settings') + [
        'entity_type' => $this->fieldDefinition->getFieldStorageDefinition()
          ->getSetting('target_type'),
      ]
    );
  }

  /**
   * Get additional fields.
   *
   * @return array|null
   *   Additional fields or NULL by default.
   */
  public function getAdditionalFields(): ?array {
    $setting = $this->getSetting('additional_fields');
    $fields = $setting['options'] ?? NULL;

    if ($setting === NULL || $fields === NULL) {
      return NULL;
    }

    return array_filter($fields);
  }

}
