<?php

namespace Drupal\Tests\entity_browser_table\Kernel;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\entity_browser_table\Plugin\Field\FieldWidget\EntityReferenceBrowserTableWidget;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\KernelTests\KernelTestBase;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\media\Entity\Media;

/**
 * Tests for the Entity Reference Browser Table widget.
 *
 * @group entity_browser_table
 */
class EntityReferenceBrowserTableWidgetTest extends KernelTestBase {

  /**
   * Test entities used for tests.
   */
  protected array $entities;

  /**
   * Field parents.
   */
  protected array $fieldParents;

  /**
   * The ID for the details' element.
   */
  protected string $detailsId;

  /**
   * The actual widget.
   */
  protected EntityReferenceBrowserTableWidget $widget;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'media',
    'image',
    'file',
    'language',
    'taxonomy',
    'text',
    'user',
    'content_translation',
    'entity_browser',
    'entity_browser_table',
  ];

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('taxonomy_term');
    $this->installConfig('media');

    // Install the United Kingdom Language.
    $language = ConfigurableLanguage::createFromLangcode('en-gb');
    $language->save();

    $fieldStorage = $this->createMock(FieldStorageConfig::class);
    $fieldStorage->method('getSetting')
      ->willReturn('media');
    $fieldDefinition = $this->createMock(FieldDefinitionInterface::class);
    $fieldDefinition->method('getFieldStorageDefinition')
      ->willReturn($fieldStorage);
    $configuration = [
      'field_definition' => $fieldDefinition,
      'settings' => [],
      'third_party_settings' => [],
    ];
    $this->widget = EntityReferenceBrowserTableWidget::create($this->container, $configuration, $this->randomString(), []);

    $this->entities = [
      Media::create([
        'bundle' => 'image',
        'name' => $this->randomString(),
        'field_brands' => [['target_id' => 1]],
      ]),
      Media::create([
        'bundle' => 'image',
        'name' => $this->randomString(),
        'field_brands' => 2,
      ]),
      Media::create([
        'bundle' => 'file',
        'name' => $this->randomString(),
        'field_brands' => 3,
      ]),
    ];
    $this->detailsId = $this->randomString();
    $this->fieldParents = [$this->randomString(), $this->randomString()];
  }

  /**
   * Test Build table rows.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function testBuildTableRows(): void {
    $testBuildTableRows = $this->widget->buildTableRows($this->detailsId, $this->fieldParents, $this->entities);
    $this->assertIsArray($testBuildTableRows);
    $this->assertCount(3, $testBuildTableRows);

    $arrayOfStrings = [
      $this->randomString(),
      $this->randomString(),
      $this->randomString(),
    ];
    $testBuildTableRows = $this->widget->buildTableRows($this->detailsId, $this->fieldParents, $arrayOfStrings);
    $this->assertIsArray($testBuildTableRows);
    $this->assertEmpty($testBuildTableRows);
  }

  /**
   * Test edit button.
   */
  public function testBuildEditButton(): void {
    $testBuildEditButton = $this->widget->buildEditButton($this->entities[0], $this->detailsId, 10, $this->fieldParents);
    $this->assertIsArray($testBuildEditButton);
    $this->assertCount(6, $testBuildEditButton);
  }

  /**
   * Test remove button.
   */
  public function testBuildRemoveButton(): void {
    $testBuildEditButton = $this->widget->buildRemoveButton($this->entities[0], $this->detailsId, 10, $this->fieldParents);
    $this->assertIsArray($testBuildEditButton);
    $this->assertCount(8, $testBuildEditButton);
  }

  /**
   * Test replace button.
   */
  public function testBuildReplaceButton(): void {
    $testBuildEditButton = $this->widget->buildReplaceButton($this->entities[0], $this->entities, $this->detailsId, 10, $this->fieldParents);
    $this->assertIsArray($testBuildEditButton);
    $this->assertCount(8, $testBuildEditButton);
  }

  /**
   * Test edit button access.
   */
  public function testGetEditButton(): void {
    $entity = $this->createMock(Media::class);
    $entity->method('access')->withAnyParameters()->willReturn(TRUE);
    $this->widget->setSetting('field_widget_edit', TRUE);
    $this->assertTrue($this->widget->getEditButtonAccess($entity));

    $entity = $this->createMock(Media::class);
    $entity->method('access')->withAnyParameters()->willReturn(FALSE);
    $this->widget->setSetting('field_widget_edit', TRUE);
    $this->assertFalse($this->widget->getEditButtonAccess($entity));
  }

  /**
   * Test replace button access.
   */
  public function testGetReplaceButtonAccess(): void {
    $entity = $this->createMock(Media::class);
    $this->widget->setSetting('field_widget_replace', TRUE);
    $this->assertTrue($this->widget->getReplaceButtonAccess([$entity]));

    $entity = $this->createMock(Media::class);
    $this->widget->setSetting('field_widget_replace', FALSE);
    $this->assertFalse($this->widget->getReplaceButtonAccess([$entity]));

    $this->widget->setSetting('field_widget_edit', TRUE);
    $this->assertFalse($this->widget->getReplaceButtonAccess($this->entities));
  }

  /**
   * Test first column function.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function testGetFirstColumn(): void {
    // @todo this never worked so needs to be re-written.
    $this->markTestSkipped();
    $this->widget->setSetting('field_widget_display', 'rendered_entity');
    $this->widget->setSetting('field_widget_display_settings', [
      "view_mode" => "preview",
    ]);
    $testGetFirstColumn = $this->widget->getFirstColumn($this->entities[0]);
    $this->assertIsArray($testGetFirstColumn);
    $this->assertArrayHasKey('#media', $testGetFirstColumn);
    $this->assertInstanceOf(Media::class, $testGetFirstColumn['#media']);
    $this->assertArrayHasKey('#view_mode', $testGetFirstColumn);
    $this->assertEquals('preview', $testGetFirstColumn['#view_mode']);

    $this->widget->setSetting('field_widget_display', 'label');
    $this->widget->setSetting('field_widget_display_settings', []);
    $testGetFirstColumn = $this->widget->getFirstColumn($this->entities[0]);
    $this->assertIsArray($testGetFirstColumn);
    $this->assertArrayHasKey('#markup', $testGetFirstColumn);
    $this->assertEquals($this->entities[0]->getName(), $testGetFirstColumn['#markup']);
  }

  /**
   * Tests additionalFieldsColumns function.
   */
  public function testAdditionalFieldsColumns(): void {
    $this->widget->setSetting('additional_fields', ['options' => ['status' => 'status']]);
    $additionalFields = $this->widget->getAdditionalFieldsColumn($this->entities[0]);
    $this->assertIsArray($additionalFields);
    $this->assertArrayHasKey('#markup', $additionalFields);
    $this->assertNotEmpty($additionalFields['#markup']);

    $this->widget->setSetting('additional_fields', ['options' => []]);
    $additionalFields = $this->widget->getAdditionalFieldsColumn($this->entities[0]);
    $this->assertEmpty($additionalFields);

    $additionalFields = $this->widget->getAdditionalFieldsColumn($this->entities[0]);
    $this->assertEmpty($additionalFields);
  }

  /**
   * Tests getFirstColumnHeader function.
   */
  public function testGetFirstColumnHeader(): void {
    $this->widget->setSetting('field_widget_display', 'rendered_entity');
    $this->assertEquals('Thumbnail', $this->widget->getFirstColumnHeader());

    $this->widget->setSetting('field_widget_display', 'label');
    $this->assertEquals('Title', $this->widget->getFirstColumnHeader());
  }

  /**
   * Test getAdditionalFieldHeader function.
   */
  public function testGetAdditionalFieldsHeader(): void {
    $this->assertNull($this->widget->getAdditionalFieldsColumnHeader());

    $this->widget->setSetting('additional_fields', ['options' => ['status' => 'status']]);
    $this->assertInstanceOf(TranslatableMarkup::class, $this->widget->getAdditionalFieldsColumnHeader());

    $this->widget->setSetting('additional_fields', ['options' => ['status' => 0]]);
    $this->assertNull($this->widget->getAdditionalFieldsColumnHeader());
  }

  /**
   * Test getActionColumnHeader function.
   */
  public function testGetActionColumnHeader(): void {
    $this->widget->setSetting('field_widget_edit', TRUE);
    $this->assertEquals('Action', $this->widget->getActionColumnHeader());

    $this->widget->setSetting('field_widget_remove', TRUE);
    $this->assertEquals('Action', $this->widget->getActionColumnHeader());

    $this->widget->setSetting('field_widget_replace', TRUE);
    $this->assertEquals('Action', $this->widget->getActionColumnHeader());

    $this->widget->setSetting('field_widget_edit', FALSE);
    $this->widget->setSetting('field_widget_remove', FALSE);
    $this->widget->setSetting('field_widget_replace', FALSE);

    $this->assertNull($this->widget->getActionColumnHeader());
  }

  /**
   * Test getAdditionalFields function.
   */
  public function testAdditionalFields(): void {
    $this->widget->setSetting('additional_fields', ['options' => []]);
    $this->assertIsArray($this->widget->getAdditionalFields());
    $this->assertEmpty($this->widget->getAdditionalFields());

    $this->widget->setSetting('additional_fields', ['options' => ['status' => 'status']]);
    $this->assertIsArray($this->widget->getAdditionalFields());
    $this->assertEquals(['status' => 'status'], $this->widget->getAdditionalFields());
  }

  /**
   * Create image media bundle.
   */
  protected function createImageMediaBundle(): void {
    $entity_type_manager = \Drupal::service('entity_type.manager');
    $media_bundle_storage = $entity_type_manager->getStorage('media_type');
    $media_bundle = $media_bundle_storage->create([
      'id' => 'image',
      'label' => 'Image',
      'description' => 'Description.',
      'type' => 'image',
      'type_configuration' => [
        'source_field' => 'field_media_image',
      ],
    ]);
    $media_bundle->save();
  }

}
