<?php

namespace Drupal\entity_bundle_scaffold\Drush;

use Consolidation\AnnotatedCommand\AnnotationData;
use Consolidation\AnnotatedCommand\Events\CustomEventAwareInterface;
use Consolidation\AnnotatedCommand\Events\CustomEventAwareTrait;
use Consolidation\AnnotatedCommand\Hooks\HookManager;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\eck\Entity\EckEntityBundle;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Completion\CompletionInput;
use Symfony\Component\Console\Completion\CompletionSuggestions;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands for creating ECK bundles.
 */
class EckBundleCreateCommands extends DrushCommands implements CustomEventAwareInterface {

  use BundleMachineNameAskTrait;
  use CustomEventAwareTrait;
  use EckEntityTypeTrait;

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
  }

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container this instance should use.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
    );
  }

  /**
   * Create a new eck entity type.
   */
  #[CLI\Command(name: 'eck:bundle:create', aliases: ['eck-bundle-create', 'ebc'])]
  #[CLI\Argument(name: 'entityType', description: 'The machine name of the entity type.')]
  #[CLI\Option(name: 'label', description: 'The human-readable name of this entity bundle.')]
  #[CLI\Option(name: 'machine-name', description: 'A unique machine-readable name for this entity bundle. It must only contain lowercase letters, numbers, and underscores.')]
  #[CLI\Option(name: 'description', description: 'Describe this entity bundle.')]
  #[CLI\Option(name: 'show-machine-names', description: 'Show machine names instead of labels in option lists.')]
  #[CLI\Usage(name: 'drush eck:bundle:create', description: 'Create an ECK entity bundle by answering the prompts.')]
  #[CLI\ValidateModulesEnabled(modules: ['eck'])]
  #[CLI\Complete(method_name_or_callable: 'complete')]
  public function createType(?string $entityType = NULL, array $options = [
    'label' => InputOption::VALUE_REQUIRED,
    'machine-name' => InputOption::VALUE_REQUIRED,
    'description' => InputOption::VALUE_OPTIONAL,
    'show-machine-names' => InputOption::VALUE_OPTIONAL,
  ]): void {
    $definition = $this->entityTypeManager->getDefinition(sprintf('%s_type', $entityType));
    $storage = $this->entityTypeManager->getStorage(sprintf('%s_type', $entityType));

    $values = [
      'status' => TRUE,
      $definition->getKey('id') => $this->input()->getOption('machine-name'),
      $definition->getKey('label') => $this->input()->getOption('label'),
      'description' => $this->input()->getOption('description') ?? '',
      'entity_type' => $entityType,
    ];

    // Command files may customize $values as desired.
    $handlers = $this->getCustomEventHandlers('eck-bundle-create');
    foreach ($handlers as $handler) {
      $handler($values);
    }

    $bundle = $storage->create($values);
    $bundle->save();

    $this->entityTypeManager->clearCachedDefinitions();
    $this->logResult($bundle);
  }

  /**
   * Fill the command options by prompting the user.
   */
  #[CLI\Hook(type: HookManager::INTERACT, target: 'eck:bundle:create')]
  public function interact(InputInterface $input, OutputInterface $output, AnnotationData $annotationData): void {
    $this->input->setArgument(
      'entityType',
      $entityTypeId = $this->input->getArgument('entityType') ?? $this->askEckEntityType()
    );
    $this->validateEckEntityType($entityTypeId);

    $this->input->setOption(
      'label',
      $this->input->getOption('label') ?? $this->askLabel()
    );
    $this->input->setOption(
      'machine-name',
      $this->input->getOption('machine-name') ?? $this->askMachineName($entityTypeId)
    );
    $this->input->setOption(
      'description',
      $this->input->getOption('description') ?? $this->askDescription()
    );
  }

  /**
   * Provide autocompletion for command arguments & options.
   */
  public function complete(CompletionInput $input, CompletionSuggestions $suggestions): void {
    if ($input->getCompletionName() === 'entityType') {
      $suggestions->suggestValues(array_keys($this->getEckEntityTypes()));
    }
  }

  /**
   * Ask for a human-readable label.
   */
  protected function askLabel(): string {
    return $this->io()->askRequired('Human-readable name');
  }

  /**
   * Ask for a description.
   */
  protected function askDescription(): ?string {
    return $this->io()->ask('Description');
  }

  /**
   * Log the command results.
   */
  private function logResult(EckEntityBundle $bundle): void {
    $this->logger()->success(
      sprintf("Successfully created %s bundle '%s'", $bundle->getEckEntityTypeMachineName(), $bundle->id())
    );

    $url = $bundle->toUrl()->setAbsolute()->toString();
    $this->logger()->success(sprintf('Further customisation can be done in the <href=%s>admin UI</>.', $url));
  }

}
