<?php

namespace Drupal\entity_bundle_scaffold\Drush;

use Consolidation\AnnotatedCommand\Events\CustomEventAwareInterface;
use Consolidation\AnnotatedCommand\Events\CustomEventAwareTrait;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\eck\Entity\EckEntityBundle;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Drush\Commands\field\EntityTypeBundleAskTrait;
use Drush\Commands\field\EntityTypeBundleValidationTrait;
use Symfony\Component\Console\Completion\CompletionInput;
use Symfony\Component\Console\Completion\CompletionSuggestions;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands for deleting ECK bundles.
 */
class EckBundleDeleteCommands extends DrushCommands implements CustomEventAwareInterface {

  use CustomEventAwareTrait;
  use EntityTypeBundleAskTrait;
  use EntityTypeBundleValidationTrait;
  use EckEntityTypeTrait;

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The entity type bundle info service.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityTypeBundleInfoInterface $entityTypeBundleInfo,
  ) {
  }

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container this instance should use.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
    );
  }

  /**
   * Delete an ECK entity type.
   */
  #[CLI\Command(name: 'eck:bundle:delete', aliases: ['eck-bundle-delete', 'ebd'])]
  #[CLI\Argument(name: 'entityType', description: 'The machine name of the entity type.')]
  #[CLI\Argument(name: 'bundle', description: 'The machine name of the bundle.')]
  #[CLI\Usage(name: 'drush eck:bundle:delete', description: 'Delete an ECK entity type by answering the prompts.')]
  #[CLI\ValidateModulesEnabled(modules: ['eck'])]
  #[CLI\Complete(method_name_or_callable: 'complete')]
  public function delete(?string $entityType = NULL, ?string $bundle = NULL): void {
    $this->input->setArgument('entityType', $entityType ??= $this->askEckEntityType());
    $this->validateEckEntityType($entityType);

    $this->input->setArgument('bundle', $bundle ??= $this->askBundle());
    $this->validateBundle($entityType, $bundle);

    $definition = $this->entityTypeManager->getDefinition(sprintf('%s_type', $entityType));
    $storage = $this->entityTypeManager->getStorage(sprintf('%s_type', $entityType));

    $bundles = $storage->loadByProperties([$definition->getKey('id') => $bundle]);
    $bundle = reset($bundles);

    // Command files may customize $values as desired.
    $handlers = $this->getCustomEventHandlers('eck-bundle-delete');
    foreach ($handlers as $handler) {
      $handler($bundle);
    }

    $storage->delete([$bundle]);

    $this->entityTypeManager->clearCachedDefinitions();
    $this->logResult($bundle);
  }

  /**
   * Provide autocompletion for command arguments & options.
   */
  public function complete(CompletionInput $input, CompletionSuggestions $suggestions): void {
    if ($input->getCompletionName() === 'entityType') {
      $suggestions->suggestValues(array_keys($this->getEckEntityTypes()));
    }

    if ($input->getCompletionName() === 'bundle') {
      $entityTypeId = $input->getArgument('entityType');
      $bundleInfo = $this->entityTypeBundleInfo->getBundleInfo($entityTypeId);

      $suggestions->suggestValues(array_keys($bundleInfo));
    }
  }

  /**
   * Log the command results.
   */
  private function logResult(EckEntityBundle $bundle): void {
    $this->logger()->success(
      sprintf("Successfully deleted %s bundle '%s'", $bundle->getEckEntityTypeMachineName(), $bundle->id())
    );
  }

}
