<?php

namespace Drupal\entity_bundle_scaffold\Drush;

use Consolidation\AnnotatedCommand\AnnotationData;
use Consolidation\AnnotatedCommand\Events\CustomEventAwareInterface;
use Consolidation\AnnotatedCommand\Events\CustomEventAwareTrait;
use Consolidation\AnnotatedCommand\Hooks\HookManager;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\eck\EckEntityTypeInterface;
use Drupal\eck\Entity\EckEntityType;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands for creating ECK entity types.
 */
class EckTypeCreateCommands extends DrushCommands implements CustomEventAwareInterface {

  use CustomEventAwareTrait;

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
  }

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container this instance should use.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
    );
  }

  /**
   * Create a new ECK entity type.
   */
  #[CLI\Command(name: 'eck:type:create', aliases: ['eck-type-create', 'etc'])]
  #[CLI\Option(name: 'label', description: 'The human-readable name of this entity bundle.')]
  #[CLI\Option(name: 'machine-name', description: 'A unique machine-readable name for this entity type. It must only contain lowercase letters, numbers, and underscores.')]
  #[CLI\Option(name: 'created', description: 'Install the created base field.')]
  #[CLI\Option(name: 'changed', description: 'Install the changed base field.')]
  #[CLI\Option(name: 'author', description: 'Install the author base field.')]
  #[CLI\Option(name: 'title', description: 'Install the title base field.')]
  #[CLI\Option(name: 'status', description: 'Install the status base field.')]
  #[CLI\Option(name: 'show-machine-names', description: 'Show machine names instead of labels in option lists.')]
  #[CLI\Usage(name: 'drush eck:type:create', description: 'Create an ECK entity type by answering the prompts.')]
  #[CLI\ValidateModulesEnabled(modules: ['eck'])]
  public function createType(array $options = [
    'label' => InputOption::VALUE_REQUIRED,
    'machine-name' => InputOption::VALUE_REQUIRED,
    'created' => InputOption::VALUE_OPTIONAL,
    'changed' => InputOption::VALUE_OPTIONAL,
    'author' => InputOption::VALUE_OPTIONAL,
    'title' => InputOption::VALUE_OPTIONAL,
    'status' => InputOption::VALUE_OPTIONAL,
    'show-machine-names' => InputOption::VALUE_OPTIONAL,
  ]): void {
    $definition = $this->entityTypeManager->getDefinition('eck_entity_type');
    $storage = $this->entityTypeManager->getStorage('eck_entity_type');

    $values = [
      'status' => $this->input()->getOption('status'),
      $definition->getKey('id') => $this->input()->getOption('machine-name'),
      $definition->getKey('label') => $this->input()->getOption('label'),
      'created' => $this->input()->getOption('created'),
      'changed' => $this->input()->getOption('changed'),
      'uid' => $this->input()->getOption('author'),
      'title' => $this->input()->getOption('title'),
    ];

    // Command files may customize $values as desired.
    $handlers = $this->getCustomEventHandlers('eck-type-create');
    foreach ($handlers as $handler) {
      $handler($values);
    }

    $type = $storage->create($values);
    $type->save();

    $this->entityTypeManager->clearCachedDefinitions();
    $this->logResult($type);
  }

  /**
   * Fill the command options by prompting the user.
   */
  #[CLI\Hook(type: HookManager::INTERACT, target: 'eck:type:create')]
  public function interact(InputInterface $input, OutputInterface $output, AnnotationData $annotationData): void {
    $this->input->setOption(
      'label',
      $this->input->getOption('label') ?? $this->askLabel()
    );
    $this->input->setOption(
      'machine-name',
      $this->input->getOption('machine-name') ?? $this->askMachineName()
    );

    foreach (['created', 'changed', 'author', 'title', 'status'] as $fieldName) {
      $this->input->setOption(
        $fieldName,
        $this->input->getOption($fieldName) ?? $this->askBaseField($fieldName)
      );
    }
  }

  /**
   * Ask for a human-readable name.
   */
  protected function askLabel() {
    return $this->io()->ask('Human-readable name');
  }

  /**
   * Ask for a machine name.
   */
  protected function askMachineName() {
    $label = $this->input->getOption('label');
    $suggestion = NULL;
    $machineName = NULL;

    if ($label) {
      $suggestion = $this->generateMachineName($label);
    }

    while (!$machineName) {
      $answer = $this->io()->ask('Machine-readable name', $suggestion);

      if (preg_match('/[^a-z0-9_]+/', (string) $answer)) {
        $this->logger()->error('The machine-readable name must contain only lowercase letters, numbers, and underscores.');
        continue;
      }

      if (strlen((string) $answer) > $this->getIdMaxLength()) {
        $this->logger()->error('The machine-readable name must not be longer than :maxLength characters.', [':maxLength' => ECK_ENTITY_ID_MAX_LENGTH]);
        continue;
      }

      if ($this->entityTypeExists($answer)) {
        $this->logger()->error('An entity type with this name already exists.');
        continue;
      }

      $machineName = $answer;
    }

    return $machineName;
  }

  /**
   * Ask whether a certain base field should be added to the entity type.
   */
  protected function askBaseField(string $fieldName): bool {
    return $this->io()->confirm(sprintf("Add the '%s' base field?", $fieldName));
  }

  /**
   * Check whether a certain entity type already exists.
   */
  protected function entityTypeExists(string $id): bool {
    try {
      $this->entityTypeManager->getDefinition($id);
    }
    catch (PluginNotFoundException) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Generate a machine name to be used as ECK entity type ID.
   */
  protected function generateMachineName(string $source): string {
    // Only lowercase alphanumeric characters and underscores.
    $machineName = preg_replace('/[^_a-z0-9]/i', '_', $source);
    // Maximum one subsequent underscore.
    $machineName = preg_replace('/_+/', '_', $machineName);
    // Only lowercase.
    $machineName = strtolower($machineName);
    // Maximum length.
    $machineName = substr($machineName, 0, $this->getIdMaxLength());

    return $machineName;
  }

  /**
   * Log the command results.
   */
  private function logResult(EckEntityType $type): void {
    $this->logger()->success(
      sprintf("Successfully created eck entity type '%s'", $type->id())
    );

    $url = $type->toUrl()->setAbsolute()->toString();
    $this->logger()->success(sprintf('Further customisation can be done in the <href=%s>admin UI</>.', $url));
  }

  /**
   * Get the maximum length for ECK entity type IDs.
   */
  protected function getIdMaxLength(): int {
    if (defined('ECK_ENTITY_ID_MAX_LENGTH')) {
      return ECK_ENTITY_ID_MAX_LENGTH;
    }

    if (property_exists(EckEntityTypeInterface::class, 'ECK_TYPE_ID_MAX_LENGTH')) {
      return EckEntityTypeInterface::ECK_ENTITY_ID_MAX_LENGTH;
    }

    return 27;
  }

}
