<?php

namespace Drupal\entity_bundle_scaffold\Drush;

use Consolidation\AnnotatedCommand\Hooks\HookManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\entity_bundle_scaffold\Service\Generator\EntityBundleClassGenerator;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Drush\Commands\field\EntityTypeBundleAskTrait;
use Drush\Commands\field\EntityTypeBundleValidationTrait;
use Symfony\Component\Console\Completion\CompletionInput;
use Symfony\Component\Console\Completion\CompletionSuggestions;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Filesystem\Filesystem;

/**
 * Drush commands for generating entity bundle classes.
 */
class EntityBundleClassCommands extends DrushCommands {

  use EntityTypeBundleAskTrait;
  use EntityTypeBundleValidationTrait;

  /**
   * The filesystem service.
   */
  protected Filesystem $fileSystem;

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The entity type bundle info service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\entity_bundle_scaffold\Service\Generator\EntityBundleClassGenerator $entityBundleClassGenerator
   *   The entity bundle class generator.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityTypeBundleInfoInterface $entityTypeBundleInfo,
    protected ConfigFactoryInterface $configFactory,
    protected ModuleHandlerInterface $moduleHandler,
    protected EntityBundleClassGenerator $entityBundleClassGenerator,
  ) {
    $this->fileSystem = new Filesystem();
  }

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container this instance should use.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('config.factory'),
      $container->get('module_handler'),
      $container->get('entity_bundle_scaffold.entity_bundle_class_generator'),
    );
  }

  /**
   * Generate an entity bundle class.
   */
  #[CLI\Command(name: 'entity:bundle-class-generate', aliases: ['entity-bundle-class-generate', 'ebcg'])]
  #[CLI\Argument(name: 'entityType', description: 'The machine name of the entity type.')]
  #[CLI\Argument(name: 'bundle', description: 'The machine name of the entity bundle.')]
  #[CLI\Option(name: 'output-module', description: 'The module in which to generate the file.')]
  #[CLI\Option(name: 'show-machine-names', description: 'Show machine names instead of labels in option lists.')]
  #[CLI\Usage(name: 'drush entity:bundle-class-generate taxonomy_term tag', description: 'Generate an entity bundle class.')]
  #[CLI\Usage(name: 'drush entity:bundle-class-generate', description: 'Generate an entity bundle class and fill in the remaining information through prompts.')]
  #[CLI\Complete(method_name_or_callable: 'complete')]
  public function generate(?string $entityType = NULL, ?string $bundle = NULL, array $options = [
    'output-module' => InputOption::VALUE_REQUIRED,
    'show-machine-names' => InputOption::VALUE_OPTIONAL,
  ]): void {
    if (empty($options['output-module'])) {
      throw new \InvalidArgumentException('You must specify an output module through --output-module or through configuration.');
    }

    $this->input->setArgument('entityType', $entityType ??= $this->askEntityType());
    $this->validateEntityType($entityType);

    $this->input->setArgument('bundle', $bundle ??= $this->askBundle());
    $this->validateBundle($entityType, $bundle);

    $definition = $this->entityTypeManager->getDefinition($entityType);
    $existingClassName = $this->entityTypeManager->getStorage($entityType)->getEntityClass($bundle);
    $hasExisting = FALSE;

    if ($existingClassName && $existingClassName !== $definition->getClass()) {
      $hasExisting = TRUE;
      $destination = (new \ReflectionClass($existingClassName))->getFileName();

      if (file_exists($destination) && !$this->io()->confirm(sprintf('%s already exists. Append to existing class?', $existingClassName), FALSE)) {
        return;
      }

      $output = $this->entityBundleClassGenerator->generateExisting($entityType, $bundle);
    }
    else {
      $destination = $this->entityBundleClassGenerator->buildEntityBundleClassPath($entityType, $bundle, $options['output-module']);
      $output = $this->entityBundleClassGenerator->generateNew($entityType, $bundle, $options['output-module']);
    }

    $this->fileSystem->remove($destination);
    $this->fileSystem->appendToFile($destination, $output);

    $this->logger()->success(
      sprintf('Successfully %s entity bundle class.', $hasExisting ? 'updated' : 'created')
    );

    if (!$this->entityBundleClassGenerator->areModelsAnnotationBased()) {
      $this->logger()->warning("Don't forget to register your bundle class through a <href=https://www.drupal.org/node/3191609>hook_entity_bundle_info_alter implementation</>, or by using the <href=https://www.drupal.org/project/entity_model>Entity Model module</>.");
    }

    // Clear cached bundle classes.
    $this->entityBundleClassGenerator->clearEntityBundleCaches();
  }

  /**
   * Set the default value for the output-module option.
   */
  #[CLI\Hook(type: HookManager::INITIALIZE, target: 'entity:bundle-class-generate')]
  public function init(): void {
    $module = $this->input->getOption('output-module');

    if (!$module) {
      $default = $this->configFactory
        ->get('entity_bundle_scaffold.settings')
        ->get('generators.bundle_class.output_module');

      $this->input->setOption('output-module', $default);
    }
  }

  /**
   * Provide autocompletion for command arguments & options.
   */
  public function complete(CompletionInput $input, CompletionSuggestions $suggestions): void {
    if ($input->getCompletionType() === CompletionInput::TYPE_ARGUMENT_VALUE) {
      if ($input->getCompletionName() === 'entityType') {
        $entityTypes = array_filter(
          $this->entityTypeManager->getDefinitions(),
          function (EntityTypeInterface $entityType) {
            return $entityType->entityClassImplements(FieldableEntityInterface::class);
          }
        );
        $suggestions->suggestValues(array_keys($entityTypes));
      }

      if ($input->getCompletionName() === 'bundle') {
        $entityTypeId = $input->getArgument('entityType');
        $bundleInfo = $this->entityTypeBundleInfo->getBundleInfo($entityTypeId);

        $suggestions->suggestValues(array_keys($bundleInfo));
      }
    }

    if ($input->getCompletionType() === CompletionInput::TYPE_OPTION_VALUE) {
      if ($input->getCompletionName() === 'output-module') {
        $modules = $this->moduleHandler->getModuleList();
        $suggestions->suggestValues(array_keys($modules));
      }
    }
  }

}
