<?php

namespace Drupal\entity_bundle_scaffold\Drush;

use Consolidation\AnnotatedCommand\AnnotationData;
use Consolidation\AnnotatedCommand\Hooks\HookManager;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Menu\MenuParentFormSelectorInterface;
use Drupal\system\Entity\Menu;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Question\ChoiceQuestion;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush hooks for integrating the menu_ui module with other commands.
 */
class MenuUiHooks extends DrushCommands {

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\Core\Menu\MenuParentFormSelectorInterface $menuParentFormSelector
   *   The menu parent form selector.
   */
  public function __construct(
    protected ModuleHandlerInterface $moduleHandler,
    protected MenuParentFormSelectorInterface $menuParentFormSelector,
  ) {
  }

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container this instance should use.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('module_handler'),
      $container->get('menu.parent_form_selector'),
    );
  }

  /**
   * Add menu_ui related options to a command.
   */
  #[CLI\Hook(type: HookManager::OPTION_HOOK, target: 'nodetype:create')]
  public function hookOption(Command $command, AnnotationData $annotationData): void {
    if (!$this->isInstalled()) {
      return;
    }

    $command->addOption(
      'menus-available',
      '',
      InputOption::VALUE_OPTIONAL,
      'The menus available to place links in for this content type.'
    );

    $command->addOption(
      'menu-default-parent',
      '',
      InputOption::VALUE_OPTIONAL,
      'The menu item to be the default parent for a new link in the content authoring form.'
    );
  }

  /**
   * Prompt for values for the menu_ui related options.
   */
  #[CLI\Hook(type: HookManager::ON_EVENT, target: 'node-type-set-options')]
  public function hookSetOptions(InputInterface $input): void {
    if (!$this->isInstalled()) {
      return;
    }

    $this->ensureOption('menus-available', [$this, 'askMenus'], TRUE);
    $menus = $this->input->getOption('menus-available');
    if (is_string($menus)) {
      $menus = explode(',', $menus);
      $this->input->setOption('menus-available', $menus);
    }

    if ($menus === []) {
      return;
    }

    $this->ensureOption('menu-default-parent', [$this, 'askDefaultParent'], TRUE);
  }

  /**
   * Assign the values for the language related options to the config array.
   */
  #[CLI\Hook(type: HookManager::ON_EVENT, target: 'nodetype-create')]
  public function hookCreate(array &$values): void {
    if (!$this->isInstalled()) {
      return;
    }

    $values['third_party_settings']['menu_ui']['available_menus'] = $this->input->getOption('menus-available');
    $values['third_party_settings']['menu_ui']['parent'] = $this->input->getOption('menu-default-parent') ?? '';
    $values['dependencies']['module'][] = 'menu_ui';
  }

  /**
   * Prompt for the menus option.
   */
  protected function askMenus(): array {
    $menus = Menu::loadMultiple();
    $choices = ['_none' => '- None -'];

    foreach ($menus as $name => $menu) {
      $label = $this->input->getOption('show-machine-names') ? $name : $menu->label();
      $choices[$name] = $label;
    }

    $question = (new ChoiceQuestion('Available menus', $choices, '_none'))
      ->setMultiselect(TRUE);

    return array_filter(
      $this->io()->askQuestion($question) ?: [],
      fn (string $value) => $value !== '_none'
    );
  }

  /**
   * Prompt for the default parent option.
   */
  protected function askDefaultParent(): string {
    $menus = $this->input->getOption('menus-available');
    $menus = array_intersect_key(Menu::loadMultiple(), array_flip($menus));

    $options = array_map(fn (Menu $menu) => $menu->label(), $menus);
    $options = $this->menuParentFormSelector->getParentSelectOptions('', $options);

    return $this->io()->choice('Default parent item', $options, 0);
  }

  /**
   * Check whether the menu_ui module is installed.
   */
  protected function isInstalled(): bool {
    return $this->moduleHandler->moduleExists('menu_ui');
  }

  /**
   * Prompt the user for the option if it's empty.
   */
  protected function ensureOption(string $name, callable $asker, bool $required): void {
    $value = $this->input->getOption($name);

    if ($value === NULL) {
      $value = $asker();
    }

    if ($required && $value === NULL) {
      throw new \InvalidArgumentException(dt('The %optionName option is required.', [
        '%optionName' => $name,
      ]));
    }

    $this->input->setOption($name, $value);
  }

}
