<?php

namespace Drupal\entity_bundle_scaffold\Drush;

use Consolidation\AnnotatedCommand\Hooks\HookManager;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush hooks for integrating the node_singles module with other commands.
 */
class NodeSinglesHooks extends DrushCommands {

  /**
   * Constructs a NodeSinglesHooks object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   */
  public function __construct(
    protected ModuleHandlerInterface $moduleHandler,
  ) {
  }

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container this instance should use.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('module_handler'),
    );
  }

  /**
   * Add node_singles related options to the command.
   */
  #[CLI\Hook(type: HookManager::OPTION_HOOK, target: 'nodetype:create')]
  public function hookOption(Command $command): void {
    if (!$this->isInstalled()) {
      return;
    }

    $command->addOption(
      'is-single',
      '',
      InputOption::VALUE_OPTIONAL,
      'Is a content type with a single entity.'
    );
  }

  /**
   * Prompt for values for the language related options.
   */
  #[CLI\Hook(type: HookManager::ON_EVENT, target: 'node-type-set-options')]
  public function hookSetOptions(): void {
    if (!$this->isInstalled()) {
      return;
    }

    $this->input->setOption(
      'is-single',
      $this->input->getOption('is-single') ?? $this->askIsSingle()
    );
  }

  /**
   * Assign the values for the language related options to the config array.
   */
  #[CLI\Hook(type: HookManager::ON_EVENT, target: 'nodetype-create')]
  public function hookCreate(array &$values): void {
    if (!$this->isInstalled()) {
      return;
    }

    $values['third_party_settings']['node_singles']['is_single'] = (int) $this->input()->getOption('is-single');
    $values['dependencies']['module'][] = 'node_singles';
  }

  /**
   * Prompt whether the node type should be a single.
   */
  protected function askIsSingle(): bool {
    return $this->io()->confirm('Content type with a single entity?', FALSE);
  }

  /**
   * Check whether the node_singles module is installed.
   */
  protected function isInstalled(): bool {
    return $this->moduleHandler->moduleExists('node_singles');
  }

}
