<?php

namespace Drupal\entity_bundle_scaffold\Drush;

use Consolidation\AnnotatedCommand\Events\CustomEventAwareInterface;
use Consolidation\AnnotatedCommand\Events\CustomEventAwareTrait;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Url;
use Drupal\node\Entity\NodeType;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Completion\CompletionInput;
use Symfony\Component\Console\Completion\CompletionSuggestions;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands for creating node types.
 */
class NodeTypeCreateCommands extends DrushCommands implements CustomEventAwareInterface {

  use BundleMachineNameAskTrait;
  use CustomEventAwareTrait;

  /**
   * Constructs a NodeTypeCreateCommands object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityFieldManagerInterface $entityFieldManager,
  ) {
  }

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container this instance should use.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
    );
  }

  /**
   * Create a new node type.
   *
   * @see \Drupal\node\NodeTypeForm
   */
  #[CLI\Command(name: 'nodetype:create', aliases: ['nodetype-create', 'ntc'])]
  #[CLI\Option(name: 'label', description: 'The human-readable name of this entity bundle.')]
  #[CLI\Option(name: 'machine-name', description: 'A unique machine-readable name for this entity bundle. It must only contain lowercase letters, numbers, and underscores.')]
  #[CLI\Option(name: 'description', description: 'Describe this entity bundle.')]
  #[CLI\Option(name: 'title-label', description: 'The label of the title field.')]
  #[CLI\Option(name: 'preview-before-submit', description: 'Preview before submitting.', suggestedValues: [DRUPAL_DISABLED, DRUPAL_OPTIONAL, DRUPAL_REQUIRED])]
  #[CLI\Option(name: 'submission-guidelines', description: 'Explanation or submission guidelines. This text will be displayed at the top of the page when creating or editing content of this type.')]
  #[CLI\Option(name: 'status', description: "The default value of the 'Published' field.")]
  #[CLI\Option(name: 'promote', description: "The default value of the 'Promoted to front page' field.")]
  #[CLI\Option(name: 'sticky', description: "The default value of the 'Sticky at top of lists' field.")]
  #[CLI\Option(name: 'create-revision', description: "The default value of the 'Create new revision' field.")]
  #[CLI\Option(name: 'display-submitted', description: 'Display author username and publish date.')]
  #[CLI\Option(name: 'show-machine-names', description: 'Show machine names instead of labels in option lists.')]
  #[CLI\Usage(name: 'drush nodetype:create', description: 'Create a node type by answering the prompts.')]
  #[CLI\ValidateModulesEnabled(modules: ['node'])]
  #[CLI\Complete(method_name_or_callable: 'complete')]
  public function createType(array $options = [
    'label' => InputOption::VALUE_REQUIRED,
    'machine-name' => InputOption::VALUE_REQUIRED,
    'description' => InputOption::VALUE_OPTIONAL,
    'title-label' => InputOption::VALUE_OPTIONAL,
    'preview-before-submit' => InputOption::VALUE_OPTIONAL,
    'submission-guidelines' => InputOption::VALUE_OPTIONAL,
    'status' => InputOption::VALUE_OPTIONAL,
    'promote' => InputOption::VALUE_OPTIONAL,
    'sticky' => InputOption::VALUE_OPTIONAL,
    'create-revision' => InputOption::VALUE_OPTIONAL,
    'display-submitted' => InputOption::VALUE_OPTIONAL,
    'show-machine-names' => InputOption::VALUE_OPTIONAL,
  ]): void {
    $this->ensureOption('label', [$this, 'askLabel'], TRUE);
    $this->ensureOption('machine-name', [$this, 'askNodeTypeMachineName'], TRUE);
    $this->ensureOption('description', [$this, 'askDescription'], FALSE);

    // Submission form settings.
    $this->ensureOption('title-label', [$this, 'askSubmissionTitleLabel'], TRUE);
    $this->ensureOption(
      'preview-before-submit',
      [$this, 'askSubmissionPreviewMode'],
      TRUE
    );
    $this->ensureOption('submission-guidelines', [$this, 'askSubmissionHelp'], FALSE);

    // Publishing options.
    $this->ensureOption('status', [$this, 'askPublished'], TRUE);
    $this->ensureOption('promote', [$this, 'askPromoted'], TRUE);
    $this->ensureOption('sticky', [$this, 'askSticky'], TRUE);
    $this->ensureOption('create-revision', [$this, 'askCreateRevision'], TRUE);

    // Display settings.
    $this->ensureOption('display-submitted', [$this, 'askDisplaySubmitted'], TRUE);

    // Command files may set additional options as desired.
    $handlers = $this->getCustomEventHandlers('node-type-set-options');
    foreach ($handlers as $handler) {
      $handler($this->input);
    }

    $bundle = $this->input()->getOption('machine-name');
    $definition = $this->entityTypeManager->getDefinition('node');
    $storage = $this->entityTypeManager->getStorage('node_type');

    $values = [
      $definition->getKey('status') => TRUE,
      $definition->getKey('bundle') => $bundle,
      'name' => $this->input()->getOption('label'),
      'description' => $this->input()->getOption('description') ?? '',
      'new_revision' => $this->input()->getOption('create-revision'),
      'help' => $this->input()->getOption('submission-guidelines') ?? '',
      'preview_mode' => $this->input()->getOption('preview-before-submit'),
      'display_submitted' => $this->input()->getOption('display-submitted'),
    ];

    // Command files may customize $values as desired.
    $handlers = $this->getCustomEventHandlers('nodetype-create');
    foreach ($handlers as $handler) {
      $handler($values);
    }

    $type = $storage->create($values);
    $type->save();

    // Update title field definition.
    $fields = $this->entityFieldManager->getFieldDefinitions('node', $bundle);
    $titleField = $fields['title'];
    $titleLabel = $this->input()->getOption('title-label');

    if ($titleLabel && $titleLabel !== $titleField->getLabel()) {
      $titleField->getConfig($bundle)
        ->setLabel($titleLabel)
        ->save();
    }

    // Update workflow options.
    foreach (['status', 'promote', 'sticky'] as $fieldName) {
      $node = $this->entityTypeManager->getStorage('node')->create(['type' => $bundle]);
      $value = (bool) $this->input()->getOption($fieldName);

      if ($node->get($fieldName)->value != $value) {
        $fields[$fieldName]
          ->getConfig($bundle)
          ->setDefaultValue($value)
          ->save();
      }
    }

    $this->entityTypeManager->clearCachedDefinitions();
    $this->logResult($type);
  }

  /**
   * Provide autocompletion for command arguments & options.
   */
  public function complete(CompletionInput $input, CompletionSuggestions $suggestions): void {
    if ($input->getCompletionType() === CompletionInput::TYPE_OPTION_VALUE) {
      if ($input->getCompletionName() === 'machine-name') {
        $label = $input->getOption('label');

        if ($label) {
          $suggestion = $this->generateMachineName($label);
          $suggestions->suggestValue($suggestion);
        }
      }

      if ($input->getCompletionName() === 'preview-before-submit') {
        $previewModes = $this->getSubmissionPreviewModes();
        $suggestions->suggestValues(array_keys($previewModes));
      }
    }
  }

  /**
   * Prompt for a machine name.
   */
  protected function askNodeTypeMachineName(): string {
    return $this->askMachineName('node');
  }

  /**
   * Prompt for a label.
   */
  protected function askLabel(): string {
    return $this->io()->askRequired('Human-readable name');
  }

  /**
   * Prompt for a description.
   */
  protected function askDescription(): ?string {
    return $this->io()->ask('Description');
  }

  /**
   * Prompt for a title field label.
   */
  protected function askSubmissionTitleLabel(): string {
    return $this->io()->ask('Title field label', 'Title');
  }

  /**
   * Prompt for the Preview before submitting option.
   */
  protected function askSubmissionPreviewMode(): int {
    $options = $this->getSubmissionPreviewModes();

    return $this->io()->choice('Preview before submitting', $options, DRUPAL_OPTIONAL);
  }

  /**
   * Prompt for an explanation or submission guidelines.
   */
  protected function askSubmissionHelp(): ?string {
    return $this->io()->ask('Explanation or submission guidelines');
  }

  /**
   * Prompt for the default value of the published field.
   */
  protected function askPublished(): bool {
    return $this->io()->confirm('Published', TRUE);
  }

  /**
   * Prompt for the default value of the promoted field.
   */
  protected function askPromoted(): bool {
    return $this->io()->confirm('Promoted to front page', TRUE);
  }

  /**
   * Prompt for the default value of the sticky field.
   */
  protected function askSticky(): bool {
    return $this->io()->confirm('Sticky at top of lists', FALSE);
  }

  /**
   * Prompt for the Create new revision option.
   */
  protected function askCreateRevision(): bool {
    return $this->io()->confirm('Create new revision', TRUE);
  }

  /**
   * Prompt for the Display author and date information option.
   */
  protected function askDisplaySubmitted(): bool {
    return $this->io()->confirm('Display author and date information', TRUE);
  }

  /**
   * Prompt the user for the option if it's empty.
   */
  protected function ensureOption(string $name, callable $asker, bool $required): void {
    $value = $this->input->getOption($name);

    if ($value === NULL) {
      $value = $asker();
    }

    if ($required && $value === NULL) {
      throw new \InvalidArgumentException(dt('The %optionName option is required.', [
        '%optionName' => $name,
      ]));
    }

    $this->input->setOption($name, $value);
  }

  /**
   * Log the command results.
   */
  protected function logResult(NodeType $type): void {
    $this->logger()->success(
      sprintf("Successfully created node type with bundle '%s'", $type->id())
    );

    $url = Url::fromRoute('entity.node_type.edit_form', ['node_type' => $type->id()])->setAbsolute()->toString();
    $this->logger()->success(sprintf('Further customisation can be done in the <href=%s>admin UI</>.', $url));
  }

  /**
   * Get the available preview modes.
   */
  protected function getSubmissionPreviewModes(): array {
    return [
      DRUPAL_DISABLED => dt('Disabled'),
      DRUPAL_OPTIONAL => dt('Optional'),
      DRUPAL_REQUIRED => dt('Required'),
    ];
  }

}
