<?php

namespace Drupal\entity_bundle_scaffold\Drush;

use Consolidation\AnnotatedCommand\Events\CustomEventAwareInterface;
use Consolidation\AnnotatedCommand\Events\CustomEventAwareTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Url;
use Drupal\taxonomy\Entity\Vocabulary;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Completion\CompletionInput;
use Symfony\Component\Console\Completion\CompletionSuggestions;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands for creating taxonomy vocabularies.
 */
class VocabularyCreateCommands extends DrushCommands implements CustomEventAwareInterface {

  use BundleMachineNameAskTrait;
  use CustomEventAwareTrait;

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
  }

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container this instance should use.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
    );
  }

  /**
   * Create a new taxonomy vocabulary.
   *
   * @see \Drupal\taxonomy\VocabularyForm
   */
  #[CLI\Command(name: 'vocabulary:create', aliases: ['vocabulary-create', 'vc'])]
  #[CLI\Option(name: 'label', description: 'The human-readable name of this vocabulary.')]
  #[CLI\Option(name: 'machine-name', description: 'A unique machine-readable name for this entity bundle. It must only contain lowercase letters, numbers, and underscores.')]
  #[CLI\Option(name: 'description', description: 'Describe this vocabulary.')]
  #[CLI\Option(name: 'show-machine-names', description: 'Show machine names instead of labels in option lists.')]
  #[CLI\Usage(name: 'drush vocabulary:create', description: 'Create a vocabulary by answering the prompts.')]
  #[CLI\ValidateModulesEnabled(modules: ['taxonomy'])]
  #[CLI\Complete(method_name_or_callable: 'complete')]
  public function createType(array $options = [
    'label' => InputOption::VALUE_REQUIRED,
    'machine-name' => InputOption::VALUE_REQUIRED,
    'description' => InputOption::VALUE_OPTIONAL,
    'show-machine-names' => InputOption::VALUE_OPTIONAL,
  ]): void {
    $this->ensureOption('label', [$this, 'askLabel'], TRUE);
    $this->ensureOption('machine-name', [$this, 'askVocabularyMachineName'], TRUE);
    $this->ensureOption('description', [$this, 'askDescription'], FALSE);

    // Command files may set additional options as desired.
    $handlers = $this->getCustomEventHandlers('vocabulary-set-options');
    foreach ($handlers as $handler) {
      $handler($this->input);
    }

    $bundle = $this->input->getOption('machine-name');
    $definition = $this->entityTypeManager->getDefinition('taxonomy_term');
    $storage = $this->entityTypeManager->getStorage('taxonomy_vocabulary');

    $values = [
      $definition->getKey('bundle') => $bundle,
      'status' => TRUE,
      'name' => $this->input()->getOption('label'),
      'description' => $this->input()->getOption('description') ?? '',
      'hierarchy' => 0,
      'weight' => 0,
    ];

    // Command files may customize $values as desired.
    $handlers = $this->getCustomEventHandlers('vocabulary-create');
    foreach ($handlers as $handler) {
      $handler($values);
    }

    $type = $storage->create($values);
    $type->save();

    $this->entityTypeManager->clearCachedDefinitions();
    $this->logResult($type);
  }

  /**
   * Provide autocompletion for command arguments & options.
   */
  public function complete(CompletionInput $input, CompletionSuggestions $suggestions): void {
    if ($input->getCompletionType() === CompletionInput::TYPE_OPTION_VALUE) {
      if ($input->getCompletionName() === 'machine-name') {
        $label = $input->getOption('label');

        if ($label) {
          $suggestion = $this->generateMachineName($label);
          $suggestions->suggestValue($suggestion);
        }
      }
    }
  }

  /**
   * Prompt for a machine name.
   */
  protected function askVocabularyMachineName(): string {
    return $this->askMachineName('taxonomy_vocabulary');
  }

  /**
   * Prompt for a label.
   */
  protected function askLabel(): string {
    return $this->io()->askRequired('Human-readable name');
  }

  /**
   * Prompt for a description.
   */
  protected function askDescription(): ?string {
    return $this->io()->ask('Description');
  }

  /**
   * Prompt the user for the option if it's empty.
   */
  protected function ensureOption(string $name, callable $asker, bool $required): void {
    $value = $this->input->getOption($name);

    if ($value === NULL) {
      $value = $asker();
    }

    if ($required && $value === NULL) {
      throw new \InvalidArgumentException(dt('The %optionName option is required.', [
        '%optionName' => $name,
      ]));
    }

    $this->input->setOption($name, $value);
  }

  /**
   * Log the command results.
   */
  protected function logResult(Vocabulary $type): void {
    $this->logger()->success(
      sprintf("Successfully created vocabulary with bundle '%s'", $type->id())
    );

    $url = Url::fromRoute('entity.taxonomy_vocabulary.edit_form', ['taxonomy_vocabulary' => $type->id()])->setAbsolute()->toString();
    $this->logger()->success(sprintf('Further customisation can be done in the <href=%s>admin UI</>.', $url));
  }

}
