<?php

namespace Drupal\entity_bundle_scaffold\Drush;

use Consolidation\AnnotatedCommand\Hooks\HookManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\entity_bundle_scaffold\Service\Generator\ControllerClassGenerator;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Drush\Commands\field\EntityTypeBundleAskTrait;
use Drush\Commands\field\EntityTypeBundleValidationTrait;
use Symfony\Component\Console\Completion\CompletionInput;
use Symfony\Component\Console\Completion\CompletionSuggestions;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Filesystem\Filesystem;

/**
 * Drush commands for generating entity bundle controllers.
 *
 * Works with the wmcontroller module.
 */
class WmControllerCommands extends DrushCommands {

  use EntityTypeBundleAskTrait;
  use EntityTypeBundleValidationTrait;

  /**
   * The filesystem service.
   */
  protected Filesystem $fileSystem;

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The entity type bundle info service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\entity_bundle_scaffold\Service\Generator\ControllerClassGenerator $controllerClassGenerator
   *   The controller class generator.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityTypeBundleInfoInterface $entityTypeBundleInfo,
    protected ConfigFactoryInterface $configFactory,
    protected ModuleHandlerInterface $moduleHandler,
    protected ControllerClassGenerator $controllerClassGenerator
  ) {
    $this->fileSystem = new Filesystem();
  }

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container this instance should use.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('config.factory'),
      $container->get('module_handler'),
      $container->get('entity_bundle_scaffold.controller_class_generator'),
    );
  }

  /**
   * Generate a wmcontroller controller.
   */
  #[CLI\Command(name: 'wmcontroller:generate', aliases: ['wmcontroller-generate', 'wmcg'])]
  #[CLI\Argument(name: 'entityType', description: 'The machine name of the entity type.')]
  #[CLI\Argument(name: 'bundle', description: 'The machine name of the entity bundle.')]
  #[CLI\Option(name: 'output-module', description: 'The module in which to generate the file.')]
  #[CLI\Option(name: 'show-machine-names', description: 'Show machine names instead of labels in option lists.')]
  #[CLI\Usage(name: 'drush entity:bundle-class-generate taxonomy_term tag', description: 'Generate a controller.')]
  #[CLI\Usage(name: 'drush entity:bundle-class-generate', description: 'Generate a controller and fill in the remaining information through prompts.')]
  #[CLI\ValidateModulesEnabled(modules: ['wmcontroller'])]
  #[CLI\Complete(method_name_or_callable: 'complete')]
  public function generateController(?string $entityType = NULL, ?string $bundle = NULL, array $options = [
    'output-module' => InputOption::VALUE_REQUIRED,
    'show-machine-names' => InputOption::VALUE_OPTIONAL,
  ]): void {
    if (empty($options['output-module'])) {
      throw new \InvalidArgumentException('You must specify an output module through --output-module or through configuration.');
    }

    $this->input->setArgument('entityType', $entityType ??= $this->askEntityType());
    $this->validateEntityType($entityType);

    $this->input->setArgument('bundle', $bundle ??= $this->askBundle());
    $this->validateBundle($entityType, $bundle);

    $className = $this->controllerClassGenerator->buildClassName($entityType, $bundle, $options['output-module']);
    $destination = $this->controllerClassGenerator->buildControllerPath($entityType, $bundle, $options['output-module']);

    try {
      new \ReflectionClass($className);

      if (file_exists($destination) && !$this->io()->confirm(sprintf('%s already exists. Replace existing class?', $className), FALSE)) {
        return;
      }
    }
    catch (\ReflectionException) {
      // Noop.
    }

    $output = $this->controllerClassGenerator->generateNew($entityType, $bundle, $options['output-module']);
    $this->fileSystem->remove($destination);
    $this->fileSystem->appendToFile($destination, $output);

    $this->logger()->success('Successfully created controller class.');
  }

  /**
   * Add a default value for the output-module option, if necessary.
   */
  #[CLI\Hook(type: HookManager::INITIALIZE, target: 'wmcontroller:generate')]
  public function init(): void {
    $module = $this->input->getOption('output-module');

    if (!$module) {
      $default = $this->configFactory
        ->get('entity_bundle_scaffold.settings')
        ->get('generators.controller.output_module');

      $this->input->setOption('output-module', $default);
    }
  }

  /**
   * Provide autocompletion for command arguments & options.
   */
  public function complete(CompletionInput $input, CompletionSuggestions $suggestions): void {
    if ($input->getCompletionType() === CompletionInput::TYPE_ARGUMENT_VALUE) {
      if ($input->getCompletionName() === 'entityType') {
        $entityTypes = array_filter(
          $this->entityTypeManager->getDefinitions(),
          function (EntityTypeInterface $entityType) {
            return $entityType->entityClassImplements(FieldableEntityInterface::class);
          }
        );
        $suggestions->suggestValues(array_keys($entityTypes));
      }

      if ($input->getCompletionName() === 'bundle') {
        $entityTypeId = $input->getArgument('entityType');
        $bundleInfo = $this->entityTypeBundleInfo->getBundleInfo($entityTypeId);

        $suggestions->suggestValues(array_keys($bundleInfo));
      }
    }

    if ($input->getCompletionType() === CompletionInput::TYPE_OPTION_VALUE) {
      if ($input->getCompletionName() === 'output-module') {
        $modules = $this->moduleHandler->getModuleList();
        $suggestions->suggestValues(array_keys($modules));
      }
    }
  }

}
